extends CharacterBody3D

@onready var anim: AnimationPlayer = $model/AnimationPlayer
@onready var model: Node3D = $model
@onready var eyes: Node3D = $mid_anchor/eyes
@onready var head_marker: Marker3D = $head_bone/head_marker
@onready var colshape_standing: CollisionShape3D = $colshape_standing
@onready var colshape_crouching: CollisionShape3D = $colshape_crouching
@onready var rhand_anchor: Node3D = $rhand_bone/anchor
@onready var sight_raycast: RayCast3D = $sight_raycast
@onready var strike_raycast: RayCast3D = $strike_raycast
@onready var audio_nodes: Array = $Audio.get_children()
@export var spawn_weapon: String = "python_357"
@export var health: int = 60

var anim_set: Dictionary = {
	"run": "unarmed_run",
	"idle": "unarmed_idle",
	"aim_focus": "aim_focus_bigpistol",
	"aim_shoot": "aim_shoot_bigpistol",
	"die": "clown_die_1"
}
var flinch_anim: String = ""
var flinch_time: float = .60

var behavior_process_ticker: float = 0.0
const FAST_PROCESS_TICK: float = 0.049
const DEFAULT_PROCESS_TICK: float = 0.09
var behavior_process_interval: float = DEFAULT_PROCESS_TICK # FAST_PROCESS_TICK

const BASE_TARGET_DISTANCE_THRESHOLD: float = 45.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var engage_distance: float = 15.0
var melee_distance_threshold: float = 7.0
var headshot_sqdist: float = 0.9

var passive_target: CharacterBody3D
@export var active_target: PhysicsBody3D
var random_remembered_target_position: Vector3
var active_target_first_health: int = 100
var last_seen_target_position: Vector3
var last_live_delta: float = 0.0
var strike_position: Vector3
var sound_position: Vector3
#var target_board: Dictionary = {}
@export var groups_i_dont_like: Array[String] = [
	"players",
	"player_minions",
	"cultists",
	"h_grunts"
]

enum top_states {
	NONE,
	IDLE,
	ALERT,
	CONTACT,
	ATTACK_SHOOT_INTENT,
	ATTACK_MELEE_INTENT,
	ATTACK_SHOOT,
	ATTACK_MELEE,
	DEAD
}
var top_state: int = top_states.IDLE
var previous_top_state: int = top_states.IDLE

var active_weapon_node: Node3D
var active_weapon_name: String
var active_weapon_ammo_left: int = 0
var active_weapon_mag_size: int = 0
var equip_delay_time: float = .8 # .2
var attack_intent_time: float = 1.30 # 2.5
@export var attack_time: float = 1.0

var sounds_heard: Array = []
var latest_sound_position: Vector3
var hearing_distance: float = BASE_TARGET_DISTANCE_THRESHOLD * 2.0

var is_dead: bool = false
var is_dying: bool = false
var is_flinching: bool = false
var is_recently_hit: bool = false
var is_equipping: bool = false
var is_aiming: bool = false
var is_shooting: bool = false
@export var is_accuracy_boost: bool = true
var is_first_contact: bool = true

var seethrough_groups: Array = ["glass_cells"]

func _ready() -> void:
	#if spawn_weapon and not active_weapon_node:
		#equip_weapon(spawn_weapon)
	
	for seethrough_group in seethrough_groups:
		for group_node in get_tree().get_nodes_in_group(seethrough_group):
			sight_raycast.add_exception(group_node)
	return

func _physics_process(delta: float) -> void:
	if $debug_label.visible:
		$debug_label.text = str(get_top_state_name())
	
	if is_dead:
		animation_control()
		return
	
	if not is_dead:
		tick_control(delta)
	
	if not self.is_on_floor():
		A.apply_gravity(self, delta)
		#if self.velocity.is_zero_approx():
	
	return

func tick_control(delta):
	behavior_process_ticker += delta
	
	if behavior_process_ticker > behavior_process_interval:
		on_behavior_process_tick()
		animation_control()
		hearing_control()
		behavior_process_ticker = 0.0
	return

func on_behavior_process_tick() -> void:
	if is_dead:
		return
	
	behavior_control()
	self.move_and_slide()
	return

func animation_control () -> void:
	if is_dead:
		A.animate(self, anim_set["die"])
		return
	
	if is_shooting:
		A.animate(self, anim_set["aim_shoot"])
		return
	
	if is_aiming:
		A.animate(self, anim_set["aim_focus"])
		return
	
	if is_equipping:
		if not anim_set.has("equip"):
			return
		A.animate(self, anim_set["equip"])
		return
	
	if A.is_moving(self):
		A.animate(self, anim_set["run"])
		return
	else:
		A.animate(self, anim_set["idle"])
		return
	return

func hearing_control():
	sounds_heard = A.check_for_sounds(self, hearing_distance)
	if not sounds_heard:
		return
	latest_sound_position = sounds_heard.pop_back().global_position
	return

func behavior_control() -> void:
	match top_state:
		top_states.NONE:
			return
		top_states.IDLE:
			# Check if active target, chill if not
			update_active_targets()
			if active_target:
				top_state_switch_to(top_states.ALERT)
			return
		top_states.ALERT:
			if active_target and is_first_contact:
				audio_switchplay($Audio.find_children("greet_*").pick_random())
				is_first_contact = false
			# Is target attackable?
			if active_target and A.is_actor_ahead_approx(self, active_target) and A.is_actor_viewable(self, sight_raycast, active_target):
				top_state_switch_to(top_states.CONTACT)
				A.face_position(self, active_target.global_position)
			return
		top_states.CONTACT:
			# Can we attack?
			if is_equipping:
				return
			
			if spawn_weapon and not active_weapon_node:
				# TEMP \/
				#is_equipping = true
				#await get_tree().create_timer(1.57).timeout
				#is_equipping = false
				# TEMP /\
				equip_weapon(spawn_weapon)
				return
			
			# How should we attack?
			## Assume shoot until I feel like implementing something else
			top_state_switch_to(top_states.ATTACK_SHOOT_INTENT)
			return
		top_states.ATTACK_SHOOT_INTENT:
			if is_aiming:
				if A.is_actor_viewable(self, sight_raycast, active_target):
					if U.coin_flip() or not last_seen_target_position:
						last_seen_target_position = active_target.global_position
					A.face_position(self, last_seen_target_position)
				return
			
			# TEMP \/
			#await get_tree().create_timer(2.03).timeout
			# TEMP /\
			
			
			is_aiming = true
			await get_tree().create_timer(attack_intent_time * .85).timeout
			
			top_state_switch_to(top_states.ATTACK_SHOOT)
			is_aiming = false
			return

		top_states.ATTACK_SHOOT:
			if is_shooting:
				return
			
			is_shooting = true
			fire_weapon()
			
			await get_tree().create_timer(attack_time).timeout
			top_state_switch_to(top_states.ALERT)
			is_shooting = false
			return
		top_states.ATTACK_MELEE_INTENT:
			return
		top_states.ATTACK_MELEE:
			return
		top_states.DEAD:
			return
	return

func equip_weapon(weapon_name: String, equip_anim: String = ""):
	var weapon_item: Dictionary = Weapons.by_name(weapon_name)
	if active_weapon_node:
		active_weapon_node.free()
	
	for c in rhand_anchor.get_children():
		c.free()
	
	if equip_anim:
		anim_set["equip"] = equip_anim
	else:
		#anim_set["equip"] = "equip_bigpistol"
		detect_weapon_params(weapon_name)
	
	is_equipping = true
	#A.animate(self, anim_set["equip"])
	active_weapon_name = weapon_name
	active_weapon_node = weapon_item["scene"].instantiate()
	active_weapon_node.actor = self
	active_weapon_node.prop_weapon = false
	active_weapon_node.suppress_aimdot = true
	#active_weapon_node.damage = int(active_weapon_node.damage * .45)
	rhand_anchor.add_child(active_weapon_node)
	
	active_weapon_ammo_left = weapon_item["mag_size"]
	active_weapon_mag_size = weapon_item["mag_size"]
	
	if active_weapon_name == "m_16_nato":
		active_weapon_node.burst_shot_limit = 1
	
	if active_weapon_node.get("finished_shot_callback"):
		active_weapon_node.finished_shot_callback = self._weapon_finished_shot
	
	await get_tree().create_timer(equip_delay_time).timeout

	if active_weapon_node.get("raycast_container"):
		for rc in active_weapon_node.raycast_container.get_children():
			rc.add_exception(self)
	elif active_weapon_node.get("raycast"):
		active_weapon_node.raycast.add_exception(self)
	
	is_equipping = false
	
	return

func update_active_targets() -> void:
	var candidates: Array = []
	for group in groups_i_dont_like:
		for actor in get_tree().get_nodes_in_group(group):
			if not actor.is_in_group("actors"):
				continue
			if actor.is_dead:
				continue
			if A.distance(self, actor) > target_distance_threshold:
				continue
			candidates.append(actor)
	
	for actor in candidates:
		if A.is_actor_viewable(self, sight_raycast, actor) and A.is_actor_ahead_approx(self, actor):
			active_target = actor
			break
	
	return


func top_state_switch_to(new_state: int) -> void:
	previous_top_state = top_state
	top_state = new_state
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3) -> void:
	if is_dead:
		return
	
	if caller and not caller.is_in_group("h_grunts"):
		active_target = caller
		A.face_position(self, active_target.global_position)
	
	var is_headshot: bool = false
	var adjusted_damage: int = damage
	if hit_type == U.hit_types.RAYCAST and hit_pos:
		var hit_dist_to_head: float = hit_pos.distance_squared_to(head_marker.global_position)
		if hit_dist_to_head < headshot_sqdist:
			is_headshot = true
			adjusted_damage = 100 + (damage * .50)
	
	health -= adjusted_damage
	
	flinch()
	#if U.coin_flip() and U.coin_flip():
		#flinch_control()
	
	if health > 0:
		audio_switchplay($Audio.find_children("pain_*").pick_random())
		
	if health < 1:
		kill_actor()
		audio_switchplay($Audio.find_children("die_*").pick_random())
		if is_headshot:
			pass
		return
	
	if not is_recently_hit:
		is_recently_hit = true
		await get_tree().create_timer(1.8).timeout
		is_recently_hit = false
	return

func fire_weapon(trigger_time: float = 0.0) -> void:
	if not active_weapon_node:
		prints(self.name, "Attempted to fire_weapon() without active_weapon_node")
		return
	
	var assumed_raycast: RayCast3D = U.get_weapon_raycast(active_weapon_node)
	var original_raycast_localrot: Vector3
	if assumed_raycast and is_accuracy_boost:
		original_raycast_localrot = assumed_raycast.rotation_degrees
		assumed_raycast.look_at(active_target.global_position)
		await get_tree().process_frame
	
	active_weapon_node.trigger_down(self)
	
	if assumed_raycast and is_accuracy_boost:
		await get_tree().process_frame
		assumed_raycast.rotation_degrees = original_raycast_localrot

	
	if trigger_time == 0.0:
		await get_tree().process_frame
	else:
		await get_tree().create_timer(trigger_time).timeout
	
	if trigger_time > -1.0:
		active_weapon_node.trigger_up()
	return

func kill_actor() -> void:
	#is_dead = true
	if is_dying or is_dead:
		return
	
	var anim_list: Array = anim.get_animation_list()
	var die_anims: Array = anim_list.filter(func (anim_name): return anim_name.contains("die"))
	#anim_set["die"] = die_anims.pick_random()
	anim_set["die"] = "clown_die_rightside_1"
	
	is_dying = true
	
	if active_weapon_node:
		active_weapon_node.visible = false
		var world_weapon: Node3D = U.spawn_world_weapon(active_weapon_node.hud_name, rhand_anchor.global_position)
		world_weapon.global_rotation = active_weapon_node.global_rotation
	is_dead = true
	
	var colshapes: Array = [colshape_standing, colshape_crouching]
	for colshape in colshapes:
		colshape.disabled = true
	return

func flinch() -> void:
	if is_flinching:
		return
	
	flinch_anim = ["flinch_lower1", "flinch_torso2"].pick_random()
	is_flinching = true
	await get_tree().create_timer(flinch_time).timeout
	is_flinching = false
	flinch_anim = ""
	return

func detect_weapon_params(weapon_name: String) -> void:
	var big_pistol_names: Array = ["python_357"]
	if big_pistol_names.has(weapon_name):
		anim_set["equip"] = "equip_bigpistol"
		anim_set["aim_focus"] = "aim_focus_bigpistol"
		anim_set["aim_shoot"] = "aim_shoot_bigpistol"
		attack_time = 4.0
	return

func recoil_kick(amount: float) -> void:
	return

#actor.apply_body_kickback($kickback_origin.global_position, knockback_force)
func apply_body_kickback(origin: Vector3, kickback_force: float) -> void:
	var kickback_direction: Vector3 = (self.global_position - origin).normalized() * kickback_force
	self.velocity += kickback_direction
	get_tree().create_tween().tween_property(self, "velocity", Vector3.ZERO, 1.8)
	return

func audio_switchplay(audionode: AudioStreamPlayer3D) -> void:
	for a in audio_nodes:
		a.stop()
	audionode.play()
	return

func get_top_state_name() -> String:
	return top_states.keys()[top_state]

func contact_alert(new_target: PhysicsBody3D) -> void:
	prints(self.name, "contact_alert() not implemented in this specific script.")
	return
