extends AnimatableBody3D

@onready var eyes: Node3D = $mid_anchor/eyes
@onready var sight_raycast: RayCast3D = $RayCast3D
@onready var anim: AnimationPlayer = $clown_freeling/AnimationPlayer
@onready var bomb_anchor: Node3D = $clown_freeling/Armature/Skeleton3D/BoneAttachment3D/bomb_anchor
@onready var bomb_path: Path3D = $bomb_path
@onready var bomb_path_follower: PathFollow3D = $bomb_path/bomb_path_follower
@onready var bomb_raycast: RayCast3D = $bomb_path/bomb_path_follower/bomb_collider/bomb_raycast
@onready var bomb_hitbox: StaticBody3D = $bomb_path/bomb_path_follower/bomb_collider/bomb_hitbox
@onready var thrown_bomb: Node3D = $bomb_path/bomb_path_follower/bomb_collider/bomb_raycast/clown_bomb_1
@onready var explosion_positions: Node3D = $bomb_path/bomb_path_follower/bomb_collider/explosion_positions

var health: int = 30
var bomb_glb: PackedScene = preload("res://Staging/Models/clown_bomb_1.glb")
var explosion_scene: PackedScene = preload("res://Scenes/explosion.tscn")
var held_bomb: Node3D = null
var visible_meshes: Array[String] = [
	"civ_body_female_neck",
	"clown_f_arms",
	"clown_f_collar",
	"clown_f_costume_1",
	"clown_f_hair_2",
	"clown_f_head_1",
	"clown_f_shoes_2"
]
@export var is_armored: bool = false
@export var drop_on_spawn: bool = true
var is_first_drop_finished: bool = false
var drop_speed: float = 0.5
const BASE_TARGET_DISTANCE_THRESHOLD: float = 45.0
const MAX_TARGET_DISTANCE_THRESHOLD: float = 70.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var melee_distance_threshold: float = 7.0
var hearing_distance: float = 30.0
var engage_distance: float = 20.0
var reaction_time: float = 0.6

enum top_states {
	IDLE,
	ALERT,
	BOMB_THROW_INTENT,
	BOMB_THROW,
	MELEE_INTENT,
	MELEE_ATTACK,
	DEAD,
	NONE
}
var top_state: int = top_states.IDLE
var previous_top_state: int = top_state
var active_target: PhysicsBody3D
var tick: float = 0.0
var tick_interval: float = 0.1
var time_since_engaged: float = 0.0

var is_bomb_equipped: bool = false
var is_equipping_bomb: bool = false
var is_activating_bomb: bool = false
var is_bomb_activated: bool = false
var is_throwing_bomb: bool = false
var is_bomb_exploding: bool = false

var bomb_throw_speed: float = .3 # Testing speed = 0.1
var deleteme: bool = false
var is_dead: bool = false

func _ready():
	anim.play("unarmed_idle")
	set_appearance()
	bomb_raycast.add_exception(self)
	bomb_hitbox.add_collision_exception_with(self)
	await get_tree().create_timer(2.0).timeout
	deleteme = true
	return

func _physics_process(delta):
	#$debug_label.text = str(top_states.keys()[top_state])
	
	if not is_first_drop_finished and drop_on_spawn:
		var collide_result: KinematicCollision3D
		collide_result = move_and_collide(Vector3(0, -drop_speed, 0))
		if collide_result:
			is_first_drop_finished = true
	
	tick += delta
	if tick >= tick_interval and deleteme:
		tick = 0.0
		behavior_tick()
		animation_control()
	
	if is_bomb_activated:
		var bomb_collider: PhysicsBody3D = bomb_raycast.get_collider()
		if bomb_collider:
			if bomb_collider.is_in_group("actors"):
				explode_bomb()
			
			#if not bomb_collider.name.contains("bomb") and bomb_collider != self:
				#explode_bomb()
	
	if is_throwing_bomb:
		if bomb_path_follower.progress_ratio > .95 and not is_bomb_exploding:
			explode_bomb()
			return
		bomb_path_follower.progress += bomb_throw_speed
	return

func set_appearance():
	for mesh in $clown_freeling/Armature/Skeleton3D.get_children():
		if not is_instance_of(mesh, MeshInstance3D):
			continue
		if mesh.name in visible_meshes:
			mesh.visible = true
		else:
			mesh.visible = false
	return

func animation_control():
	if is_dead and not anim.assigned_animation.contains("die"):
		anim.play("clown_die_1")
		return
	if is_dead:
		return
	
	if is_equipping_bomb:
		anim.play("clown_bomb_equip")
		return
	
	if is_activating_bomb:
		anim.play("clown_bomb_activate")
		return
	
	if is_throwing_bomb:
		if anim.assigned_animation == "clown_bomb_throw_1":
			return
		anim.play("clown_bomb_throw_1")
		return
	
	#########################
	anim.play("unarmed_idle")
	return

func behavior_tick():
	# IDLE
	#	Look around for things to do
	# ALERT
	#	Check conditions for a thing to do
	# BOMB_THROW_INTENT
	# BOMB_THROW
	# DEAD
	# NONE
	match top_state:
		top_states.IDLE:
			update_targets()
			if active_target:
				$Sounds/laugh_1.play()
				top_state_switch_to(top_states.ALERT)
			return
		top_states.ALERT:
			if is_throwing_bomb: return
			
			if A.is_actor_ahead_approx(self, active_target) and A.is_actor_viewable(self, sight_raycast, active_target):
				A.face_position(self, active_target.global_position, true)
				top_state_switch_to(top_states.BOMB_THROW_INTENT)
				if target_distance_threshold < MAX_TARGET_DISTANCE_THRESHOLD:
					target_distance_threshold += 5.0
			return
		top_states.BOMB_THROW_INTENT:
			# Whip out a bomb
			if is_equipping_bomb:
				return
			if not is_bomb_equipped:
				equip_bomb()
				return
			if not is_bomb_activated:
				activate_bomb()
				return
			if is_activating_bomb:
				return
			# Activate it (tick tick tick tick)
			# Stretch arm back
			top_state_switch_to(top_states.BOMB_THROW)
			return
		top_states.BOMB_THROW:
			if is_throwing_bomb:
				return
			# Throw bomb
			$telegraphed_bomb_position.global_position = active_target.global_position
			bomb_hitbox.get_node("CollisionShape3D").disabled = false
			var target_position_y_offset: float = -5.0
			$telegraphed_bomb_position.global_position.y += target_position_y_offset
			bomb_path.curve.set_point_position(1, $telegraphed_bomb_position.position)
			var last_path_index: int = bomb_path.curve.point_count - 1
			# Length of position * (ratio_of_position_to_29 * 1.25)
			var flex_amount: float = (
				bomb_path.curve.get_point_position(1).abs().length() * (bomb_path.curve.get_point_position(1).length() / 29.0)
			)
			if bomb_path.curve.get_point_position(1).abs().length() > 30.0:
				bomb_path.curve.set_point_in(last_path_index, Vector3(0.0, flex_amount * .10, -flex_amount * .15))
			else:
				bomb_path.curve.set_point_in(last_path_index, Vector3(0.0, flex_amount * .3, flex_amount * .7))
				
			$Sounds/throw_bomb.play()
			$Sounds/falling_whistle.play()
			
			unequip_bomb()
			
			thrown_bomb.visible = true
			is_throwing_bomb = true
			
			# Wait to see how it went
			#var throw_cooloff: float = 2.8
			#await get_tree().create_timer(throw_cooloff).timeout
			# Decide what to do next
			top_state_switch_to(top_states.ALERT)
			#is_throwing_bomb = false
			return
		top_states.MELEE_INTENT:
			return
		top_states.MELEE_ATTACK:
			return
		top_states.DEAD:
			return
		top_states.NONE:
			return
	return

func top_state_switch_to(new_state: int):
	previous_top_state = top_state
	top_state = new_state
	return

func update_targets():
	var player: PhysicsBody3D = U.grptop("players")
	if not player:
		return
	
	if U.distance(self, player) < target_distance_threshold:
		active_target = player
	return

func unequip_bomb():
	is_equipping_bomb = false
	is_bomb_equipped = false
	held_bomb.free()
	return

func equip_bomb():
	is_equipping_bomb = true
	#var new_bomb: Node3D = bomb_glb.instantiate()
	held_bomb = bomb_glb.instantiate()
	bomb_anchor.add_child(
		held_bomb
	)
	var bomb_equip_time: float = 1.0
	await get_tree().create_timer(bomb_equip_time).timeout
	is_bomb_equipped = true
	is_equipping_bomb = false
	return

func activate_bomb():
	is_activating_bomb = true
	var bomb_activate_time: float = 1.0
	await get_tree().create_timer(bomb_activate_time).timeout
	bomb_raycast.enabled = true
	is_bomb_activated = true
	is_activating_bomb = false
	return

func bomb_received_hit():
	explode_bomb()
	return
	
func explode_bomb():
	if is_bomb_exploding:
		return
	
	thrown_bomb.visible = false
	is_bomb_exploding = true
	
	var explosions: Array[Node3D] = [
		explosion_scene.instantiate(),
		explosion_scene.instantiate(),
		explosion_scene.instantiate()
	]
	
	var i: int = 0
	for explosion in explosions:
		Blackboard.current_world.add_child(explosion)
		explosion.global_position = explosion_positions.get_children()[i].global_position
		#explosion.blast_radius = 11.0
		explosion.raw_damage = 80
		i += 1
	
	for explosion in explosions:
		explosion.activate(self)
		await get_tree().create_timer(.1).timeout
		i += 1
	
	reset_throwing_bomb()
	return

func reset_throwing_bomb():
	bomb_path_follower.progress_ratio = 0.0
	thrown_bomb.visible = false
	bomb_hitbox.get_node("CollisionShape3D").disabled = true
	bomb_hitbox.is_hit = false
	bomb_hitbox.add_to_group("hit_takers")
	is_bomb_activated = false
	is_throwing_bomb = false
	is_bomb_exploding = false
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	if is_dead: return
	health -= damage
	if caller:
		active_target = caller
	if health < 1:
		kill_actor()
	
	return

func kill_actor():
	if is_dead:
		return
	is_dead = true
	for sound in $Sounds.get_children():
		sound.stop()
	U.random_choice($Sounds.find_children("die_*")).play()
	top_state_switch_to(top_states.DEAD)
	$CollisionShape3D.disabled = true
	return

func contact_alert(new_target: PhysicsBody3D) -> void:
	prints(self.name, "contact_alert() not implemented in this specific script.")
	return
