extends CharacterBody3D

var scripted_event_pos: Vector3

const JUMP_VELOCITY = 4.5
const BASE_MOVE_SPEED = 11.0
var move_speed: float = BASE_MOVE_SPEED
var run_speed: float = 25.0
var movement_threshold: float = 9.0
var main_group: String = "doormen"

@onready var eyes: Node3D = self.get_node("mid_anchor/eyes")
@onready var debug_label: Label3D = self.get_node("debug_label")
@onready var anim: AnimationPlayer = self.get_node("model/AnimationPlayer")
@onready var sight_raycast: RayCast3D =  self.get_node("sight_raycast")
@onready var strike_raycast: RayCast3D = self.get_node("strike_raycast")
@onready var rhand_anchor: Node3D = self.get_node("model/Armature/Skeleton3D/attachment_hand_R/rhand_anchor")
@onready var rush_hurt_marker: Marker3D = self.get_node("base_anchor/rush_hurt_marker")
@onready var nav_agent: NavigationAgent3D = self.get_node("NavigationAgent3D")
@onready var alert_fatigue: Timer = $alert_fatigue

@export var health: int = 280
@export var spawn_weapon: String
@export var melee_damage: int = 16
@export var disarm_damage: int = 8
@export var melee_recoil: float = -30.0
@export var sword_intent_time: float = .9
@export var sword_attack_speed: float = 2.0
@export var sword_damage: int = 45 # 25
var is_dead: bool = false
@export var process_soft_lock: bool = false

var active_weapon_name: String
var active_weapon_node: Node3D
var active_weapon_ammo_left: int
var active_weapon_mag_size: int
var is_equipping: bool = false
var is_aiming: bool = false
var is_attacking: bool = false
var is_melee_attacking: bool = false
var is_flinching: bool = false
var is_intent_started: bool = false

var anim_set: Dictionary = {
	"idle": "unarmed_idle",
	"run": "unarmed_run",
	"flinch": "flinch_torso2",
	"die": "die1"
}
var animation_locked: bool = false

var ouch_sounds: Array[AudioStreamPlayer3D] = []
var die_sounds: Array[AudioStreamPlayer3D] = []

var behavior_process_ticker: float = 0.0
const FAST_PROCESS_TICK: float = 0.049
const DEFAULT_PROCESS_TICK: float = 0.10
@export var behavior_process_interval: float = FAST_PROCESS_TICK

enum top_states {
	NONE,
	IDLE,
	ALERT,
	MOVING,
	GO_TO,
	ATTACK_SHOOT_INTENT,
	ATTACK_MELEE_INTENT,
	ATTACK_SHOOT,
	ATTACK_MELEE,
	ATTACK_DISARM,
	SWORD_RUSH,
	SCRIPTED_EVENT,
	DEAD
}
@export var top_state: top_states = top_states.IDLE
var previous_top_state: int = top_states.IDLE

const BASE_TARGET_DISTANCE_THRESHOLD: float = 45.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var melee_distance_threshold: float = 7.0
var caution_distance_threshold: float = melee_distance_threshold + (melee_distance_threshold * 2.1)
var passive_target: CharacterBody3D
var active_target: CharacterBody3D
var last_seen_target_position: Vector3
var strike_position: Vector3
var sound_position: Vector3
var target_pool: Array[CharacterBody3D] = []
var groups_i_dont_like: Array[String] = [
	"players"
]

@export var follow_sounds: bool = false
var is_following_sound: bool = false
var sounds_heard: Array = []

var go_to_position: Vector3 = Vector3.ZERO
var last_go_to_position: Vector3 = Vector3.ZERO
var alert_fatigue_triggered: bool = false

@export var is_wielding_sword: bool = true
var is_rush_leader: bool = false
var is_sword_rushing: bool = false
var already_rush_stabbed: Array = []
var is_other_doorman_nearby: bool = false
var target_rig_position_name: String

# ---------------------------------------------

func _ready():
	alert_fatigue.connect("timeout", _on_alert_fatigue_timeout)
	
	# Load external anims
	for external_animation in [
		{
			"name": "disarm_attack_1",
			"path": "res://Staging/Models/Animations/doorman_disarm_attack_1.anim"
		}
	]:
		anim.get_animation_library("").add_animation(
			external_animation["name"],
			load(external_animation["path"])
		)
	return

func _physics_process(delta):
	debug_label.text = top_states.keys()[top_state] + " - " + str(health)
	$debug_label2.text = self.name
	
	if process_soft_lock:
		return
	
	if not is_on_floor() and not $hitbox_pelvis.disabled:
		A.apply_gravity(self, delta)
	
	if not is_dead:
		tick_control(delta)
	
	move_and_slide()
	animation_control()
	return

func tick_control(delta):
	behavior_process_ticker += delta
	if behavior_process_ticker > behavior_process_interval:
		on_behavior_process_tick()
		behavior_process_ticker = 0.0
	return

func top_state_switch_to(new_state: int):
	previous_top_state = top_state
	top_state = new_state
	return

func animation_control():
	if animation_locked:
		return
	
	if is_dead:
		return
	
	var current_velocity: float = self.velocity.length()
	if current_velocity >= movement_threshold:
		A.animate(self, anim_set["run"])
		if current_velocity > BASE_MOVE_SPEED:
			$Sounds/footstepper.play("bootsteps")
		else:
			$Sounds/footstepper.play("silence")
		return
	
	if self.velocity.length() < movement_threshold:
		A.animate(self, anim_set["idle"])
		$Sounds/footstepper.play("silence")
		return
		
	
	return

func behavior_control():
	match top_state:
		top_states.NONE:
			return
		top_states.IDLE:
			already_rush_stabbed.clear()
			
			var targets_by_distance: Array[Dictionary] = proximity_targets(target_distance_threshold)
			if not targets_by_distance:
				# Chilling
				return
			for target in targets_by_distance:
				var is_actor_viewable: bool = A.is_actor_viewable(self, sight_raycast, target["target"])
				if is_actor_viewable:
					active_target = target["target"]
					top_state_switch_to(top_states.ALERT)
					return
				if not is_actor_viewable:
					continue
			
			if not active_target:
				# Chilling
				return
			return
		
		top_states.ALERT:
			already_rush_stabbed.clear()
			
			if not active_target or (active_target and not A.is_actor_viewable(self, sight_raycast, active_target)):
				if alert_fatigue_triggered:
					alert_fatigue_triggered = false
					top_state_switch_to(top_states.IDLE)
					return
				
				if not alert_fatigue_triggered and alert_fatigue.is_stopped():
					alert_fatigue.start()
					return
				return
			
				
			# Assuming active target
			# Decide how to attack
			# - shotgun
			#	- If has shotgun, ...
			
			## LEFTOFF
			# Working well enough
			# They'll pick a position and go to it.
			# - add damage
			
			var distance: float = A.distance(self, active_target)
			#if is_wielding_sword and distance > melee_distance_threshold:
				#top_state_switch_to(top_states.SWORD_RUSH)
				#return
			
			
			var move_position: Vector3
			if is_other_doorman_nearby:
				rig_target_position()
				if not target_rig_position_name:
					claim_open_target_position()
				move_position = get_current_target_rig_position(target_rig_position_name)
			elif not is_other_doorman_nearby and target_rig_position_name:
				unclaim_target_position(target_rig_position_name)
				move_position = active_target.global_position
			elif not is_other_doorman_nearby and not target_rig_position_name:
				move_position = active_target.global_position
			if not move_position:
				A.face_position(self, active_target.global_position)
				move_position = self.global_position + Vector3(
					randf_range(0.0, 0.9),
					0.0,
					randf_range(0.0, 0.9),
				)
			
			var is_target_armed: bool = (active_target.is_in_group("players") and active_target.get("armed_type") != U.armed_types.UNARMED)
			if (is_wielding_sword and distance >= caution_distance_threshold) or (is_wielding_sword and distance > melee_distance_threshold and is_target_armed):
				anim_set["run"] = "sword_run"
				#var run_speed = move_speed * 2.0
				A.face_position(self, active_target.global_position)
				move_nav(false, move_position, run_speed)
				return
			if not is_target_armed and is_wielding_sword and distance < caution_distance_threshold and distance > melee_distance_threshold:
				anim_set["run"] = "sword_walk"
				A.face_position(self, active_target.global_position)
				move_nav(false, move_position)
				return
			if is_wielding_sword and distance < melee_distance_threshold:
				top_state_switch_to(top_states.ATTACK_MELEE_INTENT)
				return
			
			
			return
		
		top_states.MOVING:
			return
		
		top_states.GO_TO:
			return
		
		top_states.ATTACK_SHOOT_INTENT:
			return
		
		top_states.ATTACK_MELEE_INTENT:
			if A.is_moving(self):
				A.gradual_velo_stop(self)
			
			if is_intent_started:
				return
			
			animation_locked = true
			is_intent_started = true
			A.animate(self, "sword_melee_intent_1")
			
			var is_target_armed: bool = (active_target.is_in_group("players") and active_target.get("armed_type") != U.armed_types.UNARMED)
			var intent_time_scale: float = 1.0 if not is_target_armed else .2
			await get_tree().create_timer(sword_intent_time * intent_time_scale).timeout
			if not is_dead:
				
				if is_target_armed:
					top_state_switch_to(top_states.ATTACK_DISARM)
				else:
					top_state_switch_to(top_states.ATTACK_MELEE)
			return
		
		top_states.ATTACK_SHOOT:
			return
		
		top_states.ATTACK_MELEE:
			is_intent_started = false
			if is_melee_attacking:
				return
			
			animation_locked = true #should already be true from INTENT
			is_melee_attacking = true
			A.face_position(self, active_target.global_position)
			var sword_slash_animations: Array[String] = []
			for animation_name in anim.get_animation_list():
				if animation_name.contains("sword_slash_"):
					sword_slash_animations.append(animation_name)
			var attack_anim: String = U.random_choice(sword_slash_animations)
			
			anim.speed_scale = sword_attack_speed
			A.animate(self, attack_anim)
			U.random_choice($Sounds.find_children("sword_swipe_*")).play()
			
			melee_hit_target(active_target)
			
			await get_tree().create_timer(
				anim.get_animation(attack_anim).length # * sword_attack_speed
			).timeout
			anim.speed_scale = 1.0
			
			top_state_switch_to(top_states.ALERT)
			is_melee_attacking = false
			animation_locked = false
			return
		
		top_states.ATTACK_DISARM:
			is_intent_started = false
			if is_melee_attacking:
				return
			
			animation_locked = true #should already be true from INTENT
			is_melee_attacking = true
			A.face_position(self, active_target.global_position)
			
			var attack_anim: String = "disarm_attack_1"
			
			#anim.speed_scale = sword_attack_speed
			A.animate(self, attack_anim)
			$Sounds/disarm_attack.play()
			
			disarm_hit_target(active_target, disarm_damage)
			
			await get_tree().create_timer(
				anim.get_animation(attack_anim).length # * sword_attack_speed
			).timeout
			anim.speed_scale = 1.0
			
			top_state_switch_to(top_states.ALERT)
			is_melee_attacking = false
			animation_locked = false
			return
		
		top_states.SWORD_RUSH:
			if not active_target:
				animation_locked = false
				$Sounds/footstepper.play("silence")
				top_state_switch_to(top_states.IDLE)
				return
			
			animation_locked = true
			
			is_sword_rushing = true
			A.animate(self, "sword_stab_run")
			$Sounds/footstepper.play("bootsteps")
			
			var straight_rush_distance: float = (melee_distance_threshold * 1.25)
			var target_distance: float = A.distance(self, active_target)
			
			if target_distance > straight_rush_distance:
				A.face_position(self, active_target.global_position)
			A.apply_move(self, Vector2.UP, run_speed)
			
			
			# When within a certain distance
			# save the player's position and rush the rest of the way towards
			# that position, this way the player can dodge.
			# Allow adjusting of this position to increase/decrease difficulty
			return
		
		top_states.DEAD:
			return
		
		top_states.SCRIPTED_EVENT:
			animation_locked = true
			
			is_sword_rushing = true
			A.animate(self, "sword_stab_run")
			$Sounds/footstepper.play("bootsteps")
			#A.face_position(self, get_tree().get_first_node_in_group("players").global_position)
			A.face_position(self, scripted_event_pos)
			A.apply_move(self, Vector2.UP, run_speed)
			
			var event_pos_dist: float = self.global_position.distance_to(scripted_event_pos)
			if event_pos_dist < 15.0:
				animation_locked = false
				is_sword_rushing = false
				A.gradual_velo_stop(self)
				top_state_switch_to(top_states.IDLE)
			return
	return

func sword_rush_control():
	if not is_sword_rushing:
		return
	
	var hit_takers_by_distance: Array = proximity_targets(7.0, groups_i_dont_like)
	if len(hit_takers_by_distance) < 1:
		return
	
	for hit_taker in hit_takers_by_distance:
		if hit_taker["target"] in already_rush_stabbed:
			continue
		
		melee_hit_target(hit_taker["target"], sword_damage * 2)
		already_rush_stabbed.append(hit_taker["target"])
	return

func move_nav(face_direction: bool = true, new_position: Vector3 = Vector3.ZERO, speed: float = 0.0):
	if not speed:
		speed = move_speed
	if new_position:
		nav_agent.target_position = new_position
	var next_position: Vector3 = nav_agent.get_next_path_position()
	if face_direction:
		A.face_position(self, next_position)
		A.apply_move(self, Vector2(0, -1), speed)
	elif not face_direction:
		self.velocity = (next_position - self.global_position).normalized() * speed
	return

func melee_hit_target(target: PhysicsBody3D = null, damage: int = 0):
	if not target:
		target = active_target
	if not damage:
		damage = sword_damage
	var dist: float = target.global_position.distance_to(self.global_position)
	if dist < melee_distance_threshold and A.is_actor_ahead_approx(self, target):
		target.hit(sword_damage, U.hit_types.UNDEFINED, self, rhand_anchor.global_position)
	return

func disarm_hit_target(target: PhysicsBody3D = null, damage: int = 0):
	if not target:
		target = active_target
	if not damage:
		damage = sword_damage
	var dist: float = target.global_position.distance_to(self.global_position)
	if dist < melee_distance_threshold and A.is_actor_ahead_approx(self, target):
		target.hit(sword_damage, U.hit_types.UNDEFINED, self, rhand_anchor.global_position)
		$Sounds/disarm_hit.play()
		if target.is_in_group("players") and target.get("armed_type") != U.armed_types.UNARMED:
			target.drop_hand_items()
			A.apply_move(active_target, Vector2.DOWN, 11.0)
	return

func proximity_targets(proximity: float, groups: Array[String] = []):
	var target_candidates: Array = []
	if not groups:
		groups = groups_i_dont_like
	for group in groups:
		target_candidates.append_array(get_tree().get_nodes_in_group(group))
	
	var target_distances: Dictionary = {}
	var all_distances: Array[float] = []
	for candidate in target_candidates:
		if candidate == self:
			continue
		
		if candidate in target_distances:
			continue
		
		var dist: float = self.global_position.distance_to(candidate.global_position)
		if dist < proximity:
			target_distances[dist] = {"target": candidate, "distance": dist}
			all_distances.append(dist)
	
	all_distances.sort()
	var sorted_targets: Array[Dictionary] = []
	for distance in all_distances:
		sorted_targets.append(target_distances[distance])
	
	return sorted_targets

func rig_target_position(target: Node3D = null):
	if not target:
		target = active_target
	# Check if target has rig already
	if target.has_node("relpos_rig"):
		return
	else:
		U.spawn_relpos_rig(target)
	return

func get_open_target_position(target: Node3D = null):
	# Check relpos rig for an opening
	if not target:
		target = active_target
	var rig: Node3D = target.get_node("relpos_rig")
	
	var open_position_name: String = rig.find_open_position_name()
	var open_position_vector: Vector3
	if not open_position_name:
		open_position_vector = Vector3.ZERO
		return {}
	else:
		open_position_vector = rig.get_rig_position_by_name(open_position_name)
	return {
		"name": open_position_name,
		"vector": open_position_vector
	}

func claim_open_target_position(target: Node3D = null) -> Vector3:
	var open_position: Dictionary = get_open_target_position(target)
	if open_position.is_empty():
		return Vector3.ZERO
	
	if not target:
		target = active_target
	var rig: Node3D = target.get_node("relpos_rig")
	rig.claim_position(self, open_position["name"])
	target_rig_position_name = open_position["name"]
	
	return open_position["vector"]

func unclaim_target_position(position_name: String, target: Node3D = null):
	if not target:
		target = active_target
	var rig: Node3D = target.get_node("relpos_rig")
	rig.unclaim_position(self, target_rig_position_name)
	target_rig_position_name = ""
	return

func get_current_target_rig_position(position_name: String, target: Node3D = null):
	if not target:
		target = active_target
	
	var rig: Node3D = target.get_node("relpos_rig")
	return rig.get_rig_position_by_name(position_name)

func hit(damage: int, hit_type: int, caller: CharacterBody3D, hit_pos: Vector3):
	#if is_dead: return # If no dead flinch, uncomment
	
	
	health -= damage
	if caller and not is_dead:
		var caller_distance: float = self.global_position.distance_to(caller.global_position)
		if caller_distance > target_distance_threshold:
			target_distance_threshold = caller_distance
		active_target = caller
		A.face_position(self, active_target.global_position)
	
	if health < 1:
		kill_actor()
	elif health > 0 and top_state != top_states.ATTACK_DISARM:
		animation_locked = true
		A.animate(self, anim_set["flinch"])
		await get_tree().create_timer(
			anim.get_animation(anim_set["flinch"]).length
		).timeout
		animation_locked = false
	return

func kill_actor():
	is_dead = true
	animation_locked = true
	A.animate(self, anim_set["die"])
	$Sounds/footstepper.play("silence")
	for hitbox in self.find_children("hitbox_*"):
		hitbox.disabled = true
	top_state_switch_to(top_states.DEAD)
	
	await get_tree().create_timer(3.5).timeout
	process_soft_lock = true
	return

func on_behavior_process_tick():
	if is_dead and top_state == top_states.DEAD:
		return
	
	sensory_update_control()
	sword_rush_control()
	behavior_control()
	return

func sensory_update_control():
	# If lag is noticed, only call this when needed.
	# At the moment this is called on behavior process tick
	var bump_distance: float = 5.0 # 20.0
	var doormen_by_distance: Array = proximity_targets(bump_distance, ["doormen"])
	
	is_other_doorman_nearby = len(doormen_by_distance) > 0
	
	return

func _on_alert_fatigue_timeout():
	alert_fatigue_triggered = true
	return
