extends CharacterBody3D

var main_group_name: String = "h_grunts"
const JUMP_VELOCITY: float = 4.5
const BASE_MOVE_SPEED: float = 21.0 # 16.5
var move_speed: float = BASE_MOVE_SPEED
var movement_threshold: float = 1.0
var run_scale: float = 4.5 # 3.5 # 3.0
var rope_drop_speed: float = 6.0 # 11.0
var time_since_last_moved: float = 0.0
var stuck_threshold: float = 2.3

@onready var eyes: Node3D = self.get_node("mid_anchor/eyes")
@onready var debug_label: Label3D = self.get_node("debug_label")
@onready var anim: AnimationPlayer = self.get_node("model/AnimationPlayer")
@onready var sight_raycast: RayCast3D =  self.get_node("sight_raycast")
@onready var strike_raycast: RayCast3D = self.get_node("strike_raycast")
@onready var floor_poker: RayCast3D = $strategy_rays/floor_poker
@onready var nav_agent: NavigationAgent3D = self.get_node("NavigationAgent3D")
@onready var rhand_anchor: Node3D = $rhand_bone/rhand_anchor
@onready var trigger_ready_fallback: Timer = $trigger_ready_fallback
@onready var cmd_general_cooloff: Timer = $leader_timers/cmd_general_cooloff
@onready var all_colliders: Array = [
	$CollisionShape3D,
	$crouching_colshape,
	$proning_colshape
]
@onready var head_marker: Marker3D = $head_bone/head_marker
@onready var worn_helmet_mesh: MeshInstance3D = $model/Armature/Skeleton3D/grunt_helmet
var headshot_sqdist: float = 0.9
#@onready var hitboxes: Array = [
	#$hitbox_head,
	#$hitbox_chest,
	#$hitbox_pelvis,
	#$hitbox_lower_leg_L,
	#$hitbox_lower_leg_R,
	#$hitbox_upper_leg_L,
	#$hitbox_upper_leg_R,
	#$hitbox_foot_L,
	#$hitbox_foot_R
#]
const NAVPROBE_RESTEXT: String = "res://Scenes/actor_navprobe.tscn"
var navprobe: Node3D

@export var performance_mode: bool = false

@export var health: int = 90 # 150
var spawned_health: int
var previous_health: int = health
var pre_commanded_move_rot: Vector3 = Vector3.ZERO
@export var is_leader: bool = false
@export var spawn_weapon: String = "m_16_nato"
@export var melee_damage: float = 16.0
@export var melee_recoil: float = -30.0
@export var equip_delay_time: float = 1.0
@export var alert_time: float = 0.25
@export var shoot_intent_time: float = .7
@export var attack_time: float = .80
@export var attack_delay: float = .80
var aggressive_attack_delay: float = .40
@export var rope: Node3D
@export var watcher: bool = false
var is_awaiting_alert_time: bool = false
var is_dead: bool = false
var is_dying: bool = false
var is_expired: bool = false
var is_recently_hit: bool = false
var is_target_just_spotted: bool = false
var is_hybrid_weap_prop: bool = true
var time_killed: float = 0.0
var time_low: float = 0.0
var max_time_low: float = 2.8
var max_time_killed: float = .8

var is_get_low_allowed: bool = true
var get_low_cooloff: float = 4.33
var get_low_cooloff_ticker: float = 0.0
var is_croucher: bool = false
var is_crouching: bool = false
var is_proner: bool = false
var is_moving_to_prone: bool = false
var is_prone_completed: bool = false
var is_proning: bool = false
var is_rappelling: bool = false
var is_reloading: bool = false

var active_weapon_name: String
var active_weapon_node: Node3D
var active_weapon_ammo_left: int
var active_weapon_mag_size: int
var is_trigger_ready: bool = false
var is_equipping: bool = false
var is_aiming: bool = false
var is_aiming_blocked: bool = false
var is_cover_fire_allowed: bool = true

var is_animation_blocked: bool = false
var is_aim_transitioning: bool = false
var is_attacking: bool = false
var is_melee_attacking: bool = false
var is_flinching: bool = false
var flinch_time: float = 0.5
var flinch_anim: String = ""
var is_alert_moving: bool = false
var animation_locked: bool = false

var anim_set: Dictionary = {
	"focus": "aim_focus",
	"shoot": "aim_shoot",
	"idle": "unarmed_idle",
	"crouch": "crouch_aim",
	"crouch_run": "crouch_run",
	"run": "unarmed_run",
	"sprint": "armed_sprint",
	"equip": "draw_pistol2",
	"die": "die1"
}
var anim_options: Dictionary = {
	"idle": {
		"alert": "armed_alert_idle",
		"notalert": "armed_idle"
	},
	"run": {
		"alert": "armed_aim_omni_run",
		"notalert": "armed_alert_run"
	}
}

var ouch_sounds: Array[AudioStreamPlayer3D] = []
var die_sounds: Array[AudioStreamPlayer3D] = []

var behavior_process_ticker: float = 0.0
const FAST_PROCESS_TICK: float = 0.049
const DEFAULT_PROCESS_TICK: float = 0.09
const ROPE_PROCESS_TICK: float = 0.049
var behavior_process_interval: float = DEFAULT_PROCESS_TICK # FAST_PROCESS_TICK

enum top_states {
	NONE,
	IDLE,
	ALERT,
	CONTACT,
	ATTACK_SHOOT_INTENT,
	ATTACK_MELEE_INTENT,
	ATTACK_SHOOT,
	ATTACK_MELEE,
	ROPE_STANDBY,
	ROPE_DROP_QUIET,
	ROPE_DROP_LOUD,
	ROPE_LAND,
	ROPE_DEAD,
	COMMANDED_MOVING,
	DEAD
}
var top_state: int = top_states.IDLE
var previous_top_state: int = top_states.IDLE
var default_rope_drop_state: int = top_states.ROPE_DROP_LOUD
var rope_drop_ready: bool = false
enum rope_drop_states {
	INIT,
	DROPPING,
	FINISH
}
var rope_drop_state: int = rope_drop_states.INIT



const BASE_TARGET_DISTANCE_THRESHOLD: float = 45.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var engage_distance: float = 15.0
var melee_distance_threshold: float = 7.0
var grunt_alert_distance: float = 42.0
var leader_recruit_distance: float = 75.0 # 45.0
var passive_target: CharacterBody3D
#@export var active_target: CharacterBody3D
@export var active_target: PhysicsBody3D
var random_remembered_target_position: Vector3
var active_target_first_health: int = 100
var last_seen_target_position: Vector3
var last_live_delta: float = 0.0
var strike_position: Vector3
var sound_position: Vector3
#var target_pool: Array[CharacterBody3D] = []
var target_board: Dictionary = {}
@export var groups_i_dont_like: Array[String] = [
	"players",
	"player_minions",
	"cultists"
]

@export var follow_sounds: bool = false
var is_following_sound: bool = false
var sounds_heard: Array = []
var latest_sound_position: Vector3
var hearing_distance: float = BASE_TARGET_DISTANCE_THRESHOLD * 2.0

var go_to_position: Vector3 = Vector3.ZERO
var last_go_to_position: Vector3 = Vector3.ZERO
var last_aiming_rotation: Vector3 = Vector3.ZERO

@export var is_in_world_queue: bool = false
var world_queue: Node3D
@export var soft_process_disable: bool = false
@export var fake_process_disable: bool = false
var animation_control_pause: bool = false

### GRUNT
var squad_leader: Node3D
@export var custom_squad_leader: String = ""
var commanded_position: Vector3 = Vector3.ZERO

var squad_units: Array = []
#var squad_unit_states
enum leader_command_states {
	IDLE,
	ADVANTAGE,
	DISADVANTAGE,
	RETREAT,
	PROTRACTION
}
var leader_command_state: leader_command_states = leader_command_states.IDLE
var previous_leader_command_state: int = leader_command_state

enum probe_states {
	NONE,
	IDLE,
	SETUP,
	PROBING,
	COMPLETED
}
var probe_state: int = probe_states.NONE
var previous_probe_state: int = probe_state
var probe_params: Dictionary = {}
var is_awaiting_probe_result: bool = false

enum roe_fire_modes {
	NEVER,
	HOLD,
	AT_WILL
}
enum roe_engage_modes {
	DISENGAGE,
	ENGAGE
}
var roe_fire: int = roe_fire_modes.AT_WILL
var roe_engage: int = roe_engage_modes.ENGAGE
var favored_strafe_side: Vector2 = Vector2.RIGHT
var is_first_contact: bool = true
###

func _ready() -> void:
	spawned_health = health
	
	if not active_weapon_node:
		equip_weapon(spawn_weapon)
	for window in get_tree().get_nodes_in_group("windows"):
		if window.is_in_group("breakables"):
			sight_raycast.add_exception(window)
	if U.coin_flip():
		favored_strafe_side = Vector2.LEFT
	
	if self.get_node("actor_navprobe"):
		self.get_node("actor_navprobe").actor = self
	
	if rope:
		floor_poker.enabled = true
		#behavior_process_ticker = ROPE_PROCESS_TICK
		top_state_switch_to(top_states.ROPE_STANDBY)
	
	if is_leader:
		$editor_options.gasmask = false
		load_navprobe()
		probe_state_switch_to(probe_states.IDLE)
		
		await get_tree().create_timer(.1).timeout
		recruit_nearby_grunts()
	else:
		$editor_options.gasmask = true
	return

func _physics_process(delta: float) -> void:
	if $debug_label.visible:
		var debug_leader_text: String
		if self.is_leader:
			debug_leader_text = "[L]"
		elif not self.is_leader and squad_leader:
			debug_leader_text = "-> " + squad_leader.name
		else:
			debug_leader_text = ""
		$debug_label.text = self.name + "\n" + \
			str(health) + "\n" + \
			str(top_states.keys()[top_state]) + "\n" + \
			"SPEED: " + str(move_speed) + "\n" + \
			debug_leader_text
		
		if top_states.keys()[top_state].contains("COMMAND"):
			$debug_label_squadstatus.text = "C!"
			$debug_spotlight.visible = true
		else:
			$debug_label_squadstatus.text = ""
			$debug_spotlight.visible = false
	
	
	if is_dead:
		return
	
	if is_dying:
		time_killed += delta
	
	if time_killed > max_time_killed:
		is_dead = true
		cleanup_after_death()
		return
	
	if A.is_moving(self):
		time_since_last_moved = 0.0
	else:
		time_since_last_moved += delta
	
	last_live_delta = delta
	if get_low_cooloff_ticker > 0.0:
		get_low_cooloff_ticker -= delta
		if get_low_cooloff_ticker < 0.0:
			is_get_low_allowed = true
	
	if is_crouching or is_proning:
		time_low += delta
		if time_low > max_time_low:
			standup()
			#time_low = 0.0
			#is_get_low_allowed = false
			#get_low_cooloff_ticker = get_low_cooloff
	
	
	if not is_dead:
		tick_control(delta)
	
	if not self.is_on_floor():
		A.apply_gravity(self, delta)
		#if self.velocity.is_zero_approx():
	
	return

func tick_control(delta):
	behavior_process_ticker += delta
	
	if behavior_process_ticker > behavior_process_interval:
		if not is_in_world_queue:
			on_behavior_process_tick()
			animation_control()
			hearing_control()
		behavior_process_ticker = 0.0
	return

func on_behavior_process_tick() -> void:
	if is_dead:
		return
	
	behavior_control()
	#if not self.velocity.is_zero_approx():
		#self.velocity = Vector3.ZERO
	self.move_and_slide()
	return

func behavior_control() -> void:
	match top_state:
		top_states.NONE:
			return
		top_states.IDLE:
			update_active_targets()
			
			if latest_sound_position:
				A.face_position(self, latest_sound_position)
				latest_sound_position = Vector3.ZERO
			if not active_target:
				return
			if active_target:
				top_state_switch_to(top_states.ALERT)
			return
		top_states.ALERT:
			# "What do I do now?"
			# - Check RoE
			
			if not active_target:
				top_state_switch_to(top_states.IDLE)
				return
			if active_target.is_dead:
				top_state_switch_to(top_states.IDLE)
				return
			
			anim_set["idle"] = anim_options["idle"]["alert"]
			last_seen_target_position = active_target.global_position
			if squad_leader:
				squad_leader.last_seen_target_position = self.last_seen_target_position
			
			if watcher:
				if not A.is_actor_viewable(self, sight_raycast, active_target):
					return
			A.face_position(self, active_target.global_position)
			
			top_state_switch_to(top_states.CONTACT)
			return
		top_states.CONTACT:
			if not active_target:
				top_state_switch_to(top_states.IDLE)
				return
			
			if is_first_contact:
				is_first_contact = false
				active_target_first_health = active_target.health
				if U.coin_flip() or is_leader:
					if is_leader:
						$audio/head.find_children("*contact*").pick_random().play()
					else:
						$audio/tail.find_children("*contact*").pick_random().play()
				alert_nearby_grunts()
			
			if is_leader and get_squad_hp() > 0.0:
				leader_roe_control()
			else:
				roe_control()
			return
		top_states.ATTACK_SHOOT_INTENT:
			return
		top_states.ATTACK_MELEE_INTENT:
			return
		top_states.ATTACK_SHOOT:
			return
		top_states.ATTACK_MELEE:
			return
		top_states.ROPE_STANDBY:
			if rope_drop_ready:
				top_state_switch_to(default_rope_drop_state)
			return
		top_states.ROPE_DROP_QUIET:
			return
		top_states.ROPE_DROP_LOUD:
			var control_finished: bool = rope_drop_loud_control()
			if control_finished:
				$CollisionShape3D.disabled = false
				is_rappelling = false
				if active_target:
					top_state_switch_to(top_states.ALERT)
				else:
					top_state_switch_to(top_states.IDLE)
				return
			
			return
		top_states.ROPE_LAND:
			# I guess just switch to IDLE or ALERT!
			return
		top_states.ROPE_DEAD:
			# Hang on the rope or drop
			
			return
		top_states.COMMANDED_MOVING:
			if is_crouching or is_proning:
				standup()
			
			if not commanded_position:
				prints(self.name, "COMMANDED_MOVING with no commanded_position")
				move_speed = BASE_MOVE_SPEED
				self.global_rotation = pre_commanded_move_rot
				top_state_switch_to(top_states.ALERT)
				return
			
			var position_range: float = 3.0 # 13.0
			var distance_to_position: float = self.global_position.distance_to(commanded_position)
			if distance_to_position < position_range:
				A.face_position(self, last_seen_target_position)
				commanded_position = Vector3.ZERO
				move_speed = BASE_MOVE_SPEED
				top_state_switch_to(top_states.ALERT)
				return
			
			if time_since_last_moved > stuck_threshold:
				A.face_position(self, last_seen_target_position)
				commanded_position = Vector3.ZERO
				move_speed = BASE_MOVE_SPEED
				top_state_switch_to(top_states.ALERT)
				return
			
#			move_speed = BASE_MOVE_SPEED * run_scale
			move_nav(true, commanded_position)
			if U.coin_flip():
				self.global_rotation_degrees.y += randf_range(-15.0, 15.0)
			
			#velocity.y += 3.0
			
			if A.is_actor_viewable(self, sight_raycast, active_target) and A.is_actor_ahead_approx(self, active_target):
				commanded_position = Vector3.ZERO
				move_speed = BASE_MOVE_SPEED
				self.global_rotation = pre_commanded_move_rot
				top_state_switch_to(top_states.ALERT)
			return
		top_states.DEAD:
			return
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3) -> void:
	if is_dead:
		return
	
	if caller and not caller.is_in_group("h_grunts"):
		active_target = caller
		if squad_leader:
			squad_leader.active_target = caller
		A.face_position(self, active_target.global_position)
	
	var is_headshot: bool = false
	var adjusted_damage: int = damage
	if hit_type == U.hit_types.RAYCAST and hit_pos:
		var hit_dist_to_head: float = hit_pos.distance_squared_to(head_marker.global_position)
		if hit_dist_to_head < headshot_sqdist:
			is_headshot = true
			adjusted_damage = 100 + (damage * .50)
	
	health -= adjusted_damage
	
	flinch()
	#if U.coin_flip() and U.coin_flip():
		#flinch_control()
	
	if health < 1:
		kill_actor()
		if is_headshot:
			pop_helmet()
		return
	
	if not is_recently_hit:
		is_recently_hit = true
		await get_tree().create_timer(1.8).timeout
		is_recently_hit = false
	return

func kill_actor() -> void:
	#is_dead = true
	if is_dying or is_dead:
		return
	
	for audio in $audio.find_children("*", "AudioStreamPlayer3D"):
		audio.stop()
	
	if is_leader:
		$audio/head.find_children("*pain*").pick_random().play()
	else:
		$audio/tail.find_children("*pain*").pick_random().play()
	is_dying = true
	if not is_rappelling:
		top_state_switch_to(top_states.DEAD)
	active_weapon_node.visible = false
	var world_weapon: Node3D = U.spawn_world_weapon(active_weapon_node.hud_name, rhand_anchor.global_position)
	world_weapon.global_rotation = active_weapon_node.global_rotation
	return

func top_state_switch_to(new_state: int) -> void:
	previous_top_state = top_state
	top_state = new_state
	state_switch_hook()
	return

func leader_command_state_switch_to(new_state: int) -> void:
	previous_leader_command_state = leader_command_state
	leader_command_state = new_state
	return

func probe_state_switch_to(new_state: int) -> void:
	previous_probe_state = probe_state
	probe_state = new_state
	return

func flinch() -> void:
	if is_flinching:
		return
	
	flinch_anim = ["flinch_lower1", "flinch_torso2"].pick_random()
	is_flinching = true
	await get_tree().create_timer(flinch_time).timeout
	is_flinching = false
	flinch_anim = ""
	return

func animation_control() -> void:
	if is_dead:
		return
	
	if is_dying and is_rappelling:
		A.animate(self, "rope_die_1")
		return
	
	if is_dying:
		is_animation_blocked = false
		A.animate(self, anim_set["die"])
		return
	
	if is_animation_blocked:
		return
	
	if is_rappelling:
		if top_state == top_states.ROPE_DROP_LOUD:
			A.animate(self, "rope_cone_fire")
		else:
			A.animate(self, "rope_idle")
		return
	
	if is_flinching and flinch_anim:
		A.animate(self, flinch_anim)
		return
	
	if is_reloading:
		var reload_anim: String = "armedlong_reload_1" if not (is_crouching or is_proning) else "armedlong_crouch_reload"
		A.animate(self, reload_anim)
		return
	
	if is_crouching and not is_aiming and A.is_moving(self):
		A.animate(self, anim_set["crouch_run"])
		return
	
	if is_crouching and is_aiming:
		A.animate(self, anim_set["crouch"])
		return
	
	if is_proning and is_aiming:
		if not is_prone_completed and not is_moving_to_prone:
			is_animation_blocked = true
			is_moving_to_prone = true
			A.animate(self, "stand_to_prone")
			await get_tree().create_timer(anim.get_animation("stand_to_prone").length).timeout
			#A.animate(self, "prone_aim")
			is_prone_completed = true
			is_moving_to_prone = false
			is_animation_blocked = false
		
		A.animate(self, "prone_aim")
		return
	
	if (is_aim_transitioning or is_aiming) and not is_aiming_blocked:
		if A.is_moving(self):
			#A.animate(self, "armed_aim_omni_run")
			#if move_speed >= (move_speed * run_scale):
			if move_speed > BASE_MOVE_SPEED:
				A.animate(self, anim_set["sprint"])
			else:
				A.animate(self, anim_set["run"])
		else:
			A.animate(self, anim_set["focus"])
		return
	
	
	if A.is_moving(self):
		A.animate(self, anim_set["run"])
	else:
		if not is_crouching and not is_proning:
			A.animate(self, anim_set["idle"])
	return

func equip_weapon(weapon_name: String, equip_anim: String = "draw_pistol2"):
	var weapon_item: Dictionary = Weapons.by_name(weapon_name)
	if active_weapon_node:
		active_weapon_node.free()
	
	for c in rhand_anchor.get_children():
		c.free()
	
	is_equipping = true
	# A.animate(self, anim_set["equip"])
	active_weapon_name = weapon_name
	active_weapon_node = weapon_item["scene"].instantiate()
	active_weapon_node.actor = self
	active_weapon_node.prop_weapon = false
	active_weapon_node.suppress_aimdot = true
	active_weapon_node.damage = int(active_weapon_node.damage * .45)
	rhand_anchor.add_child(active_weapon_node)
	
	active_weapon_ammo_left = weapon_item["mag_size"]
	active_weapon_mag_size = weapon_item["mag_size"]
	
	if active_weapon_name == "m_16_nato":
		active_weapon_node.burst_shot_limit = 1
	
	if active_weapon_node.get("finished_shot_callback"):
		active_weapon_node.finished_shot_callback = self._weapon_finished_shot
	
	await get_tree().create_timer(equip_delay_time).timeout

	if active_weapon_node.get("raycast_container"):
		for rc in active_weapon_node.raycast_container.get_children():
			rc.add_exception(self)
	elif active_weapon_node.get("raycast"):
		active_weapon_node.raycast.add_exception(self)
	
	is_equipping = false
	
#var anim_set: Dictionary = {
	#"focus": "aim_focus",
	#"shoot": "aim_shoot",
	#"idle": "unarmed_idle",
	#"crouch": "crouch_aim",
	#"run": "unarmed_run",
	#"equip": "draw_pistol2",
	#"die": "die1"
#}
#var anim_options: Dictionary = {
	#"idle": {
		#"alert": "armed_alert_idle",
		#"notalert": "armed_idle"
	#},
	#"run": {
		#"alert": "armed_alert_run",
		#"notalert": "armed_alert_run"
	#}
#}
	if active_weapon_node.size == U.weapon_sizes.PISTOL:
		anim_set["focus"] = "aim_pistol_idle"
		anim_set["shoot"] = "aim_pistol_idle"
		anim_set["idle"] = anim_options["idle"]["notalert"]
		anim_set["run"] = "armed_aim_pistol_omni_run"
		anim_set["sprint"] = "armed_sprint_pistol"
		anim_set["crouch"] = "crouch_pistol_aim"
		#anim_set["crouch_run"] = ""
	else:
		anim_set["idle"] = anim_options["idle"]["notalert"]
		anim_set["run"] = anim_options["run"]["alert"]
	return

func update_active_targets() -> void:
	var candidates: Array = []
	for group in groups_i_dont_like:
		for actor in get_tree().get_nodes_in_group(group):
			if not actor.is_in_group("actors"):
				continue
			if actor.is_dead:
				continue
			if A.distance(self, actor) > target_distance_threshold:
				continue
			candidates.append(actor)
	
	for actor in candidates:
		if A.is_actor_viewable(self, sight_raycast, actor) and A.is_actor_ahead_approx(self, actor):
			active_target = actor
			if squad_leader:
				squad_leader.active_target = actor
			break
	
	return

func leader_roe_control() -> void:
	# Conditions -> Plans -> Actions
	# (simple though, stupid simple)
	# Example:
	# - Player is >90% health, squad is >90% health, player is greater than engage distance, less than target distance
	#	-- No orders (just do what they normally do)
	# - Player is <90% health, squad is >90% health, player is greater than engage distance, less than target distance
	#	-- MOVE UP! Find cover NW, N, NE or move toward player
	# - Player is <90% health, squad is >90% health, player is less than engage distance
	#	-- FAN OUT! Find cover W, E or move away from leader
	# - Player is <90% health, squad is <90% health, player is greater than engage distance, less than target distance
	#	-- FALL BACK! Find cover SW, S, SE or move toward leader
	# - Player hasn't been seen in 8.0s.
	#	-- SMOKE! Throw a smoke grenade
	#	-- MOVE! Go towards last player location one unit at a time
	if not active_target:
		prints(self.name, "leader_roe_control called with no active_target")
		return
	
	leader_survival_control()
	
	match leader_command_state:
		leader_command_states.IDLE:
			var target_hp: float = float(active_target.health) / float(active_target_first_health)
			var squad_hp: float = get_squad_hp()
			
			var hp_threshold: float = .90 # .85
			if target_hp > hp_threshold and squad_hp > hp_threshold:
				leader_command_state_switch_to(leader_command_states.ADVANTAGE) # Put here as a placeholder but actually, technically it is an advantage because the squad size is bigger than the player (1 or almost 1)
				return
			if target_hp < hp_threshold and squad_hp > hp_threshold:
				leader_command_state_switch_to(leader_command_states.ADVANTAGE)
				return
			if target_hp > hp_threshold and squad_hp < hp_threshold:
				leader_command_state_switch_to(leader_command_states.DISADVANTAGE)
				return
			if target_hp < hp_threshold and squad_hp < hp_threshold:
				#leader_command_state_switch_to(leader_command_states.RETREAT)
				leader_command_state_switch_to(leader_command_states.DISADVANTAGE)
				return
			return
		leader_command_states.ADVANTAGE:
			
			# If commanded to MOVE UP in the last 5.0 seconds, don't command again
			if not cmd_general_cooloff.is_stopped():
				return
			
			if probe_state != probe_states.NONE and probe_state != probe_states.IDLE and probe_state != probe_states.COMPLETED:
				probe_control()
				return
			
			if probe_state == probe_states.COMPLETED:
				# IF cover identified:
				# - tell unit to navigate to that cover position
				cmd_general_cooloff.start()
				
				if U.coin_flip():
					$audio/head.find_children("*move*").pick_random().play()
				else:
					$audio/head.find_children("*cover_fire*").pick_random().play()
				
				if probe_params["result"] == "OK":
					probe_params["unit"].squad_command({
						"command": "move",
						"position": probe_params["cover_position"]
					})
				elif probe_params["result"] == "NO":
					probe_params["unit"].squad_command({
						"command": "move",
						"position": active_target.global_position
					})
				# ELIF not cover identified:
				# - If no probe result (no good cover positions) use navagent to move unit towards the player
				# Command other grunts to shoot as cover fire for moving grunt
				
				for unit in squad_units:
					if unit.is_dead:
						continue
					if unit == probe_params["unit"]:
						continue
					unit.is_cover_fire_allowed = true
				
				reset_navprobe()
				probe_state_switch_to(probe_states.IDLE)
				leader_command_state_switch_to(leader_command_states.IDLE)
				return
			
			if probe_state == probe_states.IDLE:
				var commanded_unit: PhysicsBody3D
				
				
				## PICK CLOSEST _ START \/
				#var closest_unit_and_dist: Dictionary = {"unit": null, "dist": 0.0}
				#for unit in squad_units:
					#if unit.is_dead:
						#continue
					#var distance_to_unit: float = unit.global_position.distance_squared_to(self.global_position)
					#if closest_unit_and_dist["dist"] == 0.0:
						#closest_unit_and_dist["unit"] = unit
						#closest_unit_and_dist["dist"] = distance_to_unit
						#continue
					#if distance_to_unit < closest_unit_and_dist["dist"]:
						#closest_unit_and_dist["unit"] = unit
						#closest_unit_and_dist["dist"] = distance_to_unit
						#continue
				#if closest_unit_and_dist["dist"] == 0.0:
					#prints(self.name, "WARN: closest unit distance is 0.0 after looping through units. Unit array size:", squad_units.size())
					#return
				#
				#commanded_unit = closest_unit_and_dist["unit"]
				## PICK CLOSEST _ END /\
				
				#commanded_unit = squad_units.pick_random()
				var unit_candidates: Array = []
				for unit in squad_units:
					if not unit.is_dead:
						unit_candidates.append(unit)
				commanded_unit = unit_candidates.pick_random()
				
				probe_params.merge({
					"unit": commanded_unit,
					"target": active_target,
					#"directions": [Vector2.UP, Vector2.UP + Vector2.LEFT, Vector2.UP + Vector2.RIGHT]
					"directions": [Vector2.UP, Vector2.LEFT, Vector2.DOWN, Vector2.RIGHT]
				}, true)
				# navprobe -> ballprobe
				probe_state_switch_to(probe_states.SETUP)
				return
			
			return
		
		leader_command_states.DISADVANTAGE:
			
			# If commanded to MOVE UP in the last 5.0 seconds, don't command again
			if not cmd_general_cooloff.is_stopped():
				return
			
			if probe_state != probe_states.NONE and probe_state != probe_states.IDLE and probe_state != probe_states.COMPLETED:
				probe_control()
				return
			
			if probe_state == probe_states.COMPLETED:
				# IF cover identified:
				# - tell unit to navigate to that cover position
				cmd_general_cooloff.start()
				
				if U.coin_flip():
					$audio/head.find_children("*move*").pick_random().play()
				else:
					$audio/head.find_children("*cover_fire*").pick_random().play()
				
				if probe_params["result"] == "OK":
					probe_params["unit"].squad_command({
						"command": "move",
						"position": probe_params["cover_position"]
					})
				elif probe_params["result"] == "NO":
					probe_params["unit"].squad_command({
						"command": "move",
						"position": active_target.global_position
					})
				# ELIF not cover identified:
				# - If no probe result (no good cover positions) use navagent to move unit towards the player
				# Command other grunts to shoot as cover fire for moving grunt
				
				reset_navprobe()
				probe_state_switch_to(probe_states.IDLE)
				
				return
			
			if probe_state == probe_states.IDLE:
				var commanded_unit: PhysicsBody3D
				## PICK CLOSEST _ START \/
				#var closest_unit_and_dist: Dictionary = {"unit": null, "dist": 0.0}
				#for unit in squad_units:
					#if unit.is_dead:
						#continue
					#var distance_to_unit: float = unit.global_position.distance_squared_to(self.global_position)
					#if closest_unit_and_dist["dist"] == 0.0:
						#closest_unit_and_dist["unit"] = unit
						#closest_unit_and_dist["dist"] = distance_to_unit
						#continue
					#if distance_to_unit < closest_unit_and_dist["dist"]:
						#closest_unit_and_dist["unit"] = unit
						#closest_unit_and_dist["dist"] = distance_to_unit
						#continue
				#if closest_unit_and_dist["dist"] == 0.0:
					#prints(self.name, "WARN: closest unit distance is 0.0 after looping through units. Unit array size:", squad_units.size())
					#return
				#
				#commanded_unit = closest_unit_and_dist["unit"]
				## PICK CLOSEST _ END /\
				
				#commanded_unit = squad_units.pick_random()
				var unit_candidates: Array = []
				for unit in squad_units:
					if not unit.is_dead:
						unit_candidates.append(unit)
				commanded_unit = unit_candidates.pick_random()
				
				probe_params.merge({
					"unit": commanded_unit,
					"target": active_target,
					"directions": [Vector2.DOWN, Vector2.DOWN + Vector2.LEFT, Vector2.DOWN + Vector2.RIGHT]
				}, true)
				# navprobe -> ballprobe
				probe_state_switch_to(probe_states.SETUP)
				return
			
			
			return
		leader_command_states.RETREAT:
			return
		leader_command_states.PROTRACTION:
			return
	
	return

func roe_control() -> void:
#enum roe_fire_modes {
	#NEVER,
	#HOLD,
	#AT_WILL
#}
#enum roe_engage_modes {
	#DISENGAGE,
	#ENGAGE
#}
#var roe_fire: int = roe_fire_modes.AT_WILL
#var roe_engage: int = roe_engage_modes.ENGAGE
	
	# RoE - FIRE
	## if never fire, don't fire
	## if hold fire, wait for player to attack, then fire
	## if at will, fire whenever there's line of sight
	
	# RoE - ENGAGE
	## if disengage, keep perimiter
	## if engage, follow target until they're dead
	
	if not active_target:
		prints(self.name, "roe_control, no target")
		return
	
	roe_fire_control()
	roe_engage_control()
	return

func roe_fire_control() -> void:
	match roe_fire:
		roe_fire_modes.NEVER:
			pass
		roe_fire_modes.HOLD:
			pass
		roe_fire_modes.AT_WILL:
			if is_reloading:
				return
			
			if is_flinching:
				return
			
			if not active_weapon_node:
				return
			
			if not is_trigger_ready and trigger_ready_fallback.is_stopped():
				trigger_ready_fallback.start()
			
			if not is_aiming and not is_aim_transitioning:
				is_aim_transitioning = true
				await get_tree().create_timer(anim.get_animation(anim_set["focus"]).length).timeout
				is_aiming = true
				is_aim_transitioning = false
				is_trigger_ready = true
				return
			
			#A.face_position(self, active_target.global_position)
			var pf_is_actor_viewable: bool = A.is_actor_viewable(self, sight_raycast, active_target)
			var weap_collider: Object = active_weapon_node.raycast.get_collider()
			if pf_is_actor_viewable:
				last_seen_target_position = active_target.global_position
			
			if active_weapon_ammo_left < 1:
				pass
			
			if is_trigger_ready and weap_collider:
				if weap_collider.is_in_group("h_grunts"):
					if active_weapon_node.get("is_shooting"):
						active_weapon_node.trigger_up()
					is_aiming = false
					is_aiming_blocked = true
					var trig_prev: bool = is_trigger_ready
					is_trigger_ready = false
					await get_tree().create_timer(attack_time * 2.0).timeout
					is_trigger_ready = trig_prev
					is_aiming_blocked = false
					return
				
				if active_weapon_ammo_left < 1:
					reload_weapon()
					return
				
				if is_aiming_blocked or is_moving_to_prone:
					return
				
				if not pf_is_actor_viewable:
					if U.coin_flip() and U.coin_flip() and U.coin_flip():
						update_active_targets()
					
					#if is_target_just_spotted and not is_aiming_blocked:
					if is_cover_fire_allowed and last_seen_target_position and not is_aiming_blocked and (U.coin_flip() and U.coin_flip()):
						A.face_position(self, last_seen_target_position)
						strike_raycast.rotation = sight_raycast.rotation
						is_trigger_ready = false
						await get_tree().create_timer(attack_delay).timeout
						fire_weapon(true)
						
						return
					
					if (U.coin_flip() and U.coin_flip() and U.coin_flip() and U.coin_flip()): # and not is_proning and not is_crouching:
						if last_seen_target_position:
							A.face_position(self, last_seen_target_position)
						self.global_rotation_degrees.y += randf_range(-15.0, 15.0)
					return
				
				is_trigger_ready = false
				A.face_position(self, active_target.global_position)
				strike_raycast.rotation = sight_raycast.rotation
				#last_seen_target_position = active_target.global_position
				is_target_just_spotted = true
				get_tree().create_tween().tween_callback(
					func ():
						if not self:
							return
						is_target_just_spotted = false
				).set_delay(0.7)
				
				#await get_tree().create_timer(.1).timeout
				if is_leader:
					#if U.coin_flip():
					$audio/head.find_children("*attacking*").pick_random().play()
				await get_tree().create_timer(attack_delay).timeout
				await fire_weapon(true)
			else:
				pass
				#if is_aiming and last_seen_target_position and not pf_is_actor_viewable:
					##if U.coin_flip():
					#A.face_position(self, last_seen_target_position)
					#fire_weapon()
				#if not is_trigger_ready:
					#if pf_is_actor_viewable and not is_aiming_blocked:
						#is_trigger_ready = true
						#trigger_ready_fallback.stop()
						#return
				
	return

func roe_engage_control() -> void:
	match roe_engage:
		roe_engage_modes.DISENGAGE:
			return
		roe_engage_modes.ENGAGE:
			if is_recently_hit and U.coin_flip() and U.coin_flip():
				if is_crouching or is_proning:
					standup()
				
				var hit_move_vec: Vector2 = Vector2.DOWN
				if health < 50:
					hit_move_vec += (favored_strafe_side * randf_range(1.1, 3.3))
				else:
					hit_move_vec += (favored_strafe_side * randf_range(-1.1, 1.1))
				A.apply_move(self, hit_move_vec, move_speed)
				return
			
			if is_croucher and is_get_low_allowed:
				is_crouching = true
				set_crouching_colshape()
				A.gradual_velo_stop(self)
				
				if is_aiming:
					return
				#return
			
			if is_proner and is_get_low_allowed:
				is_proning = true
				set_proning_colshape()
				A.gradual_velo_stop(self)
				return
			
			
			
			if (U.coin_flip() and U.coin_flip() and U.coin_flip()) and not is_croucher:
				is_croucher = true
				return
			elif not is_croucher and not is_proner and (U.coin_flip() and U.coin_flip()):
				is_proner = true
				return
			
			if U.coin_flip() and U.coin_flip():
				if $strategy_rays/low_right.is_colliding() and $strategy_rays/low_left.is_colliding():
					favored_strafe_side = Vector2.ZERO
				if favored_strafe_side == Vector2.RIGHT and $strategy_rays/low_right.is_colliding():
					favored_strafe_side = Vector2.LEFT
				if favored_strafe_side == Vector2.LEFT and $strategy_rays/low_left.is_colliding():
					favored_strafe_side = Vector2.RIGHT
			
			if favored_strafe_side and U.coin_flip():
				var forward_amount: Vector2 = (Vector2.UP * randf_range(-1.25, 1.25))
				A.apply_move(self, forward_amount + favored_strafe_side, move_speed)
			else:
				var forward_amount: Vector2 = Vector2.UP if health > 75 else Vector2.DOWN
				A.apply_move(self, Vector2.UP, move_speed * 1.5)
			
			
			
			return
	return

func cleanup_after_death () -> void:
	if active_weapon_node != null:
		active_weapon_node.queue_free()
	$CollisionShape3D.disabled = true
	$crouching_colshape.disabled = true
	$proning_colshape.disabled = true
	return

func alert_nearby_grunts() -> void:
	for grunt in get_tree().get_nodes_in_group("h_grunts"):
		if A.distance(self, grunt) < grunt_alert_distance:
			grunt.active_target = self.active_target
	return

func reload_weapon() -> void:
	if is_reloading:
		return
	
	is_reloading = true
	if active_weapon_node.get("is_shooting"):
		active_weapon_node.trigger_up()
	
	$audio/generic_reload.play()
	active_weapon_node.reload(self)
	var reload_anim: String = "armedlong_reload_1" if not (is_crouching or is_proning) else "armedlong_crouch_reload"
	await get_tree().create_timer(
		anim.get_animation(reload_anim).length
	).timeout
	active_weapon_ammo_left = active_weapon_mag_size
	is_reloading = false
	return

func set_standing_colshape() -> void:
	$CollisionShape3D.disabled = false
	$crouching_colshape.disabled = true
	$proning_colshape.disabled = true
	return
func set_crouching_colshape() -> void:
	$CollisionShape3D.disabled = true
	$crouching_colshape.disabled = false
	$proning_colshape.disabled = true
	return
func set_proning_colshape() -> void:
	$CollisionShape3D.disabled = true
	$crouching_colshape.disabled = true
	$proning_colshape.disabled = false
	return

func standup() -> void:
	is_crouching = false
	is_proning = false
	is_croucher = false
	is_proner = false
	set_standing_colshape()
	time_low = 0.0
	is_get_low_allowed = false
	get_low_cooloff_ticker = get_low_cooloff
	return

func hearing_control():
	if watcher and not is_first_contact:
		return
	
	if is_aiming and top_state == top_states.CONTACT:
		return
	
	sounds_heard = A.check_for_sounds(self, hearing_distance)
	if not sounds_heard:
		return
	latest_sound_position = sounds_heard.pop_back().global_position
	return

func fire_weapon(force_trigger: bool = false, ff_check: bool = true, retard_check: bool = true) -> void:
	if not is_trigger_ready and not force_trigger:
		return
	if not active_weapon_node:
		return
	if is_dead or is_dying:
		return
	
	if active_weapon_ammo_left < 1:
		return
	
	if ff_check:
		var is_trigger_safe: bool = true
		var sight_collider: Object = sight_raycast.get_collider()
		var weap_collider: Object = active_weapon_node.raycast.get_collider()
		if sight_collider:
			if sight_collider.is_in_group("h_grunts"):
				is_trigger_safe = false
		if weap_collider:
			if weap_collider.is_in_group("h_grunts"):
				is_trigger_safe = false
		if not is_trigger_safe:
			return
	
	if retard_check and active_target:
		var is_collision_too_close: bool = false
		var col_prox_threshold: float = 10.0
		var sight_collider: Object = sight_raycast.get_collider()
		var weap_collider: Object = active_weapon_node.raycast.get_collider()
		if sight_collider and sight_collider != active_target:
			if sight_raycast.get_collision_point().distance_to(self.global_position) < col_prox_threshold:
				is_collision_too_close = true
		if weap_collider and weap_collider != active_target:
			if active_weapon_node.raycast.get_collision_point().distance_to(self.global_position) < col_prox_threshold:
				is_collision_too_close = true
		
		if is_collision_too_close:
			return
	
	var y_diff_threshold: float = 13.0 # 3.0 # 4.0 # 3.0
	var y_diff: float = absf(active_target.global_position.y - self.global_position.y)
	var original_weapon_raycast: RayCast3D = active_weapon_node.raycast
	
	
	if y_diff > y_diff_threshold:
		active_weapon_node.prop_weapon = true
	
	active_weapon_node.trigger_up()
	active_weapon_node.trigger_down(self)
	
	
	if active_weapon_node.name != "m_16_nato" and active_weapon_node.fire_mode == U.fire_modes.FULL_AUTO:
		await get_tree().create_timer(randf_range(0.2, 0.63)).timeout
	
	if not active_weapon_node:
		return
	
	if y_diff > y_diff_threshold:
		#sight_raycast.rotation_degrees.x += 15.0
		var impact_raycast: RayCast3D = sight_raycast # strike_raycast
		var rc_g_rot: Vector3 = impact_raycast.global_rotation
		var impact_collider: Object = impact_raycast.get_collider()
		if impact_collider:
			A.face_position(self, impact_collider.global_position)
		impact_raycast.rotation_degrees.x += 15.0
		A.create_impact(self, impact_raycast, active_weapon_node.damage, active_weapon_node.knockback_force)
		active_weapon_node.prop_weapon = false
		impact_raycast.rotation_degrees.x -= 15.0
	
	if not active_weapon_node.get("finished_shot_callback"):
		active_weapon_ammo_left -= 1
	
	await get_tree().create_timer(randf_range(0, attack_time * 1.5)).timeout
	is_trigger_ready = true
	return


func _on_trigger_ready_fallback_timeout() -> void:
	is_trigger_ready = true
	if active_weapon_node != null:
		active_weapon_node.trigger_up()
	return

func rope_drop_loud_control() -> bool:
	var control_finished: bool = false
	
	match rope_drop_state:
		rope_drop_states.INIT:
			floor_poker.enabled = false
			for col in all_colliders:
				col.disabled = true
			
			self.global_position = rope.get_start_pos()
			
			is_trigger_ready = true
			
			is_rappelling = true
			rope.drop_rope()
			rope_drop_state = rope_drop_states.DROPPING
			get_tree().create_tween().tween_callback(
				func ():
					if not self:
						return
					$CollisionShape3D.disabled = false
					floor_poker.enabled = true
			).set_delay(.9)
			return control_finished
		rope_drop_states.DROPPING:
			# Descend
			var dir_to_rope: Vector3 = (rope.get_end_pos() - self.global_position).normalized()
			if health > 0:
				self.velocity += dir_to_rope * rope_drop_speed
			elif health < 1:
				A.apply_gravity(self, last_live_delta)
			# After like .5 seconds, enable colliders
			# if LOUD:
			#	start shooting, downward cone (easy, just shoot with the weapon raycast and let the anim handle it)
			if is_trigger_ready:
				is_trigger_ready = false
				active_weapon_node.trigger_down(self)
				get_tree().create_tween().tween_callback(
					func ():
						if not self:
							return
						if not is_rappelling:
							return
						if not active_weapon_node:
							return
						active_weapon_node.trigger_up()
						is_trigger_ready = true
				).set_delay(randf_range(.1, .4))
			
			# When end of rope or downward collider detected, switch to land
			if floor_poker.is_colliding():
				rope_drop_state = rope_drop_states.FINISH
			return control_finished
		rope_drop_states.FINISH:
			if active_weapon_node:
				active_weapon_node.trigger_up()
			update_active_targets()
			if not active_target:
				active_target = get_tree().get_first_node_in_group("players")
			if active_target:
				A.face_position(self, active_target.global_position)
			control_finished = true
			return control_finished

	return control_finished

func recoil_kick(recoil_amount: float) -> void:
	return

func _weapon_finished_shot(caller: Object) -> void:
	active_weapon_ammo_left -= 1
	return

func _weapon_action_triggered(caller: Object) -> void:
	active_weapon_ammo_left -= 1
	return

func recruit_nearby_grunts() -> void:
	for grunt in get_tree().get_nodes_in_group("h_grunts"):
		if grunt.is_leader:
			continue
		if grunt.is_dead:
			continue
		if grunt == self:
			continue
		if A.distance(self, grunt) > leader_recruit_distance:
			continue
		
		grunt.set_squad_leader(self)
		grunt.is_cover_fire_allowed = false
		squad_units.append(grunt)
	return

func load_navprobe() -> void:
	navprobe = load(NAVPROBE_RESTEXT).instantiate()
	navprobe.actor = self
	self.add_child(navprobe) # Not needed since I'm assigning to a prop?
	navprobe.visible = $debug_label.visible
	return

func get_squad_hp() -> float:
	# total_grunt_health / total_grunt_spawn_health
	var total_grunt_health: int = 0
	var total_grunt_spawn_health: int = 0
	for unit in squad_units:
		if unit.health > 0:
			total_grunt_health += unit.health
		total_grunt_spawn_health += unit.spawned_health
	var squad_hp: float = float(total_grunt_health) / float(total_grunt_spawn_health)
	return squad_hp


func probe_control() -> void:
#func execute_probe() -> void:
	#probe_state = probe_states.SETUP
	#navprobe.global_position = commanded_unit.global_position
	#navprobe
	## Get first good cover position from ballprobe
	#return
	
	match probe_state:
		probe_states.NONE:
			return
		probe_states.IDLE:
			return
		probe_states.SETUP:
			# Check we have the params
			# - unit (move to position, rotate with unit)
			# - target (active_target)
			# - directions (Vector2's for ball launch)
			var param_check_OK: bool = true
			for param in ["unit", "target", "directions"]:
				if not probe_params.has(param):
					param_check_OK = false
					break
				if probe_params[param] == null:
					param_check_OK = false
					break
			
			if param_check_OK:
			# Move navprobe to unit
			# Rotate navprobe with unit
				navprobe.global_position = probe_params["unit"].global_position
				navprobe.global_rotation = probe_params["unit"].global_rotation
				probe_state_switch_to(probe_states.PROBING)
				return
			if not param_check_OK:
				probe_params["result"] = "param check failed"
				probe_state_switch_to(probe_states.COMPLETED)
				return
			
			return
		probe_states.PROBING:
#####
	#var navprobe: Node3D = get_tree().get_first_node_in_group("navprobes")
	#var ballprobe_results: Dictionary = await navprobe.launch_ballprobes([
		#Vector2.UP,
		#Vector2.UP + Vector2.RIGHT,
		#Vector2.UP + Vector2.LEFT
	#], self)
	#prints(self.name, "RES:")
	#for probe_res in ballprobe_results.keys():
		#prints(ballprobe_results[probe_res] == self)
	#
#####
			if is_awaiting_probe_result:
				return
			
			is_awaiting_probe_result = true
			var probe_result: Dictionary = await navprobe.launch_ballprobes(probe_params["directions"], active_target)
			navprobe.launch_speed = navprobe.launch_speed + (navprobe.launch_speed * .25) # * .10)
			navprobe.ball_mass += 0.01
			if not probe_result:
				prints(self.name, "No probe result:", probe_result)
				print_stack()
				push_error("Unexpected empty dictionary on probe_result")
				return
			
			# Get cover position, tell the leader state and exit probing
			var cover_position: Vector3 = Vector3.ZERO
			var cover_position_candidates: Array = []
			for probe in probe_result.keys():
				var ray_collider: Object = probe_result[probe]
				if not ray_collider:
					prints(self.name, "Probe collided with nothing when casting towards target.")
					continue
				if ray_collider == active_target:
					prints(self.name, "Probe collided with target, not a cover position")
					continue
				if ray_collider != active_target:
					#cover_position = probe.global_position
					#break
					cover_position_candidates.append(probe.global_position)
			
			if cover_position_candidates:
				cover_position = cover_position_candidates.pick_random()
			
			probe_params["cover_position"] = cover_position
			if cover_position:
				probe_params["result"] = "OK"
			else:
				probe_params["result"] = "NO"
			probe_state_switch_to(probe_states.COMPLETED)
			is_awaiting_probe_result = false
			await get_tree().create_timer(1.0).timeout
			for probe in probe_result.keys():
				probe.queue_free()
			return
		probe_states.COMPLETED:
			if probe_params["result"] == "param check failed":
				prints(self.name, "PARAM CHECK FAILED ON PROBE! :(")
				push_error("PARAM CHECK FAILED ON PROBE!")
				return
			return
	
	return


func _on_cmd_general_cooloff_timeout() -> void:
	return

func reset_navprobe() -> void:
	navprobe.position = Vector3.ZERO
	navprobe.rotation = Vector3.ZERO
	return

func squad_command(cmd_params: Dictionary) -> void:
	if cmd_params["command"] == "move":
		commanded_position = cmd_params["position"]
		time_since_last_moved = 0.0
		move_speed = BASE_MOVE_SPEED * run_scale
		pre_commanded_move_rot = self.global_rotation
		$leader_timers/move_timeout.start()
		top_state_switch_to(top_states.COMMANDED_MOVING)
		return
	return

func move_nav(face_direction: bool = true, new_position: Vector3 = Vector3.ZERO):
	
	if new_position:
		nav_agent.target_position = new_position
	var next_position: Vector3 = nav_agent.get_next_path_position()
	if $debug_label.visible:
		U.spawn_green_marker(Blackboard.current_world, nav_agent.target_position, .5)
		U.spawn_blue_marker(Blackboard.current_world, next_position, .5)
	if face_direction:
		A.face_position(self, next_position)
		A.apply_move(self, Vector2(0, -1), move_speed)
	elif not face_direction:
		self.velocity = (next_position - self.global_position).normalized() * move_speed
	return

func leader_survival_control() -> void:
	if not active_target:
		return
	
	is_croucher = U.coin_flip()
	
	if is_croucher and is_get_low_allowed:
		is_crouching = true
		set_crouching_colshape()
		A.gradual_velo_stop(self)
	
	if last_seen_target_position and U.coin_flip():
		A.face_position(self, last_seen_target_position)
	
	if A.is_actor_viewable(self, sight_raycast, active_target) and A.is_actor_ahead_approx(self, active_target):
		if active_weapon_ammo_left < 1:
			reload_weapon()
			return
		
		if not is_aiming and not is_aim_transitioning:
			is_aim_transitioning = true
			await get_tree().create_timer(anim.get_animation(anim_set["focus"]).length).timeout
			is_aiming = true
			is_aim_transitioning = false
			is_trigger_ready = true
			return
		
		
		if is_trigger_ready:
			await get_tree().create_timer(attack_delay * 2.0).timeout
			await fire_weapon(true)
			await get_tree().create_timer(attack_time).timeout
			if active_weapon_node != null:
				active_weapon_node.trigger_up()
	
	return


func _on_move_timeout_timeout() -> void:
	if top_state == top_states.COMMANDED_MOVING:
		top_state_switch_to(top_states.ALERT)
		return
	
	return

func set_squad_leader(caller: Node3D) -> void:
	if custom_squad_leader:
		if caller.name == custom_squad_leader:
			self.squad_leader = caller
	elif not custom_squad_leader:
		self.squad_leader = caller
	return

func pop_helmet() -> void:
	#var helmet: RigidBody3D = $effects/helmet_pop/phys_helmet
	var helmet: RigidBody3D = self.get_node("effects/helmet_pop/phys_helmet")
	if helmet == null:
		prints(self.name, "tried to pop helmet but helmet was null")
		return
	helmet.reparent(Blackboard.current_world)
	worn_helmet_mesh.visible = false
	helmet.visible = true
	helmet.add_collision_exception_with(self)
	helmet.get_node("CollisionShape3D").disabled = false
	helmet.process_mode = Node.PROCESS_MODE_INHERIT
	helmet.apply_central_impulse(
		Vector3(0.0, 16.5, 15.0).rotated(Vector3.UP, self.global_rotation.y)
	)
	#helmet.apply_torque_impulse(Vector3(.5, 0.0, 0.0))
	return

func play_special_audio(audio_name: String) -> void:
	$audio/special_event.get_node(audio_name).play()
	return

func state_switch_hook() -> void:
	if active_target:
		for a in $audio/special_event.get_children():
			a.stop()
	return
