extends AnimatableBody3D

@export var health: int = 60
@export var spawn_weapon: String = "beretta_nine"
var weapon_name: String
var dbg_node_name: String
@export var equip_on_spawn: bool = true
@export var is_turret: bool = false

@onready var anim: AnimationPlayer = $model/AnimationPlayer
@onready var eyes: Node3D = $mid_anchor/eyes
@onready var sight_raycast: RayCast3D = $Vision/RayCast3D
@onready var umarker: Marker3D = $UtilityMarker
@onready var collision_ray_container: Node3D = $collision_ray_container
@onready var floor_sensor: RayCast3D = $floor_sensor
@onready var rhand_anchor: Node3D = $Bones/rhand_attachment/anchor
@onready var umove_expire: Timer = $umove_expire
const reject_world_queue: bool = true

var move_speed: float = 1.0
var gravity_speed: float = 1.0
var previous_umove_position: Vector3 = Vector3.ZERO
var umove_conversion_scale: float = 0.1 # Originally .1

enum top_states {
	IDLE,
	ALERT,
	ATTACK_SHOOT_INTENT,
	ATTACK_SHOOT,
	ATTACK_MELEE_INTENT,
	ATTACK_MELEE,
	MOVE_DIRECTION,
	MOVE_DESTINATION,
	DEAD,
	NONE
}
@export var top_state: top_states = top_states.IDLE
var previous_top_state: top_states = top_state

enum move_modes {
	DESTINATION,
	DIRECTION,
	NONE
}
var move_mode: move_modes = move_modes.NONE
var move_destination: Vector3
var move_direction: Vector2
var is_move_blind: bool = false
var travel_distance: float = 0.0

var ticker: float = 0.0
const BASE_TICKER_INTERVAL: float = 0.049
const FAST_TICKER_INTERVAL: float = 0.010
var ticker_interval: float = 0.049
var active_weapon: Node3D
var active_weapon_last_raycast_rotation: Vector3
var is_using_weapon_raycast: bool = true
var active_target: Node3D
var passive_target: Node3D
var last_seen_target_position: Vector3
var sounds_heard: Array = []
var target_board: Dictionary = {}
var groups_i_dont_like: Array[String] = [
	"players",
	"player_minions",
	"cultists",
	"wiseguys",
	"punks"
]
const BASE_TARGET_DISTANCE_THRESHOLD: float = 45.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var engage_distance: float = 15.0
var hearing_distance: float = 50.0
var melee_distance_threshold: float = 7.0
var shoot_intent_time: float = .8
var shoot_duration: float = .8
var shoot_cooloff: float = .5
var is_dead: bool = false
var is_post_death: bool = false
var is_running: bool = false
var is_aiming: bool = false
var is_attacking: bool = false
var is_flinching: bool = false
var is_ever_seen_target: bool = false

var anim_set: Dictionary = {
	"idle": "unarmed_idle",
	"run": "unarmed_run",
	"flinch": "flinch_torso2",
	"die": "die_lower"
}

###
var time: float = 0.0
var last_attack_time: float = -1.0

func _ready() -> void:
	dbg_node_name = self.name
	sight_raycast.add_exception(self)
	#await get_tree().create_timer(1.0).timeout
	if spawn_weapon and equip_on_spawn:
		weapon_name = spawn_weapon
		equip_weapon(weapon_name)
	return


#func _process(delta: float) -> void:
#	return

func _physics_process(delta: float) -> void:
	if $debug_label_1.visible:
		$debug_label_1.text = str(top_states.keys()[top_state])
		$debug_label_2.text = self.name
	
	if is_dead:
		animation_control()
		return
	
	
	time += delta
	ticker += delta
	if ticker >= ticker_interval:
		hearing_control()
		behavior_control()
		animation_control()
		ticker = 0.0
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	if process_mode == Node.PROCESS_MODE_DISABLED: return
	if self.get_parent().process_mode == Node.PROCESS_MODE_DISABLED: return
	health -= damage
	if health < 1:
		kill_actor()
		return
	else:
		flinch()
	
	if caller and not is_attacking:
		A.face_position(self, caller.global_position)
	return

func kill_actor():
	if is_dead:
		return
	if active_weapon and not is_attacking:
		active_weapon.visible = false
	anim_set["die"] = [
		"die1",
		"die2",
		"die_lower",
		"die_head3",
		"die_pistol_1",
		"die3"
	].pick_random()
	
	is_dead = true
	for c in collision_ray_container.get_children():
		c.enabled = false
	if active_weapon:
		if active_weapon.get("raycast"):
			active_weapon.raycast.enabled = false
	sight_raycast.enabled = false
	$CollisionShape3D.disabled = true
	$colshape_half.disabled = true
	
	if active_weapon and not is_attacking:
		#U.spawn_world_weapon(active_weapon.name, rhand_anchor.global_position)
		U.spawn_world_weapon(active_weapon.hud_name, rhand_anchor.global_position)
		for c in rhand_anchor.get_children():
			if is_attacking or is_aiming:
				break
			#c.visible = false
			c.queue_free()
	
	top_state_switch_to(top_states.DEAD)
	return

func top_state_switch_to(new_state: int):
#	if self.name == "wiseguy": # DEBUG
#		prints(top_states.keys()[top_state], " -> ", top_states.keys()[new_state])
	previous_top_state = top_state
	top_state = new_state
	return

func behavior_control():
	if dbg_node_name == "mook_lite_5":
		pass
	match top_state:
		top_states.IDLE:
			update_active_target()
			
			if not active_target:
				return
			
			if A.is_actor_ahead_approx(self, active_target) and A.is_actor_viewable(self, sight_raycast, active_target):
				is_ever_seen_target = true
				top_state_switch_to(top_states.ALERT)
				return
			
			if not is_ever_seen_target:
				return
			return
		top_states.ALERT:
			if weapon_name and not active_weapon:
				equip_weapon(weapon_name)
			elif not weapon_name:
				# Add code here to look for a weapon and/or melee
				return
			
			if is_target_attackable():
				top_state_switch_to(top_states.ATTACK_SHOOT_INTENT)
			elif A.is_actor_viewable(self, sight_raycast, active_target):
				A.face_position(self, active_target.global_position)
			elif get_seconds_since_last_attack() > 1.2 and U.coin_flip() and not is_turret:
				var random_move_direction: Vector2 = Vector2(
					float(randi_range(-1, 1)),
					float(randi_range(-1, 1))
				)
				var move_distance: float = randf_range(2.0, 10.0) # 10.0
				switch_to_move_state({
					"mode": move_modes.DIRECTION,
					"vector": random_move_direction,
					"distance": move_distance
				})
			return
		top_states.ATTACK_SHOOT_INTENT:
			if is_aiming:
				return
			
			if get_seconds_since_last_attack() < shoot_cooloff and not is_turret:
				var random_move_direction: Vector2 = Vector2(
					float(randi_range(-1, 1)),
					float(randi_range(-1, 1))
				)
				switch_to_move_state({
					"mode": move_modes.DIRECTION,
					"vector": random_move_direction,
					"distance": 10.0
				})
				return
			
			# Laugh and point gun
			A.face_position(self, active_target.global_position)
			if U.coin_flip():
				[$Sounds/laugh_1, $Sounds/laugh_2].pick_random().play()
			is_aiming = true
			await get_tree().create_timer(shoot_intent_time).timeout
			top_state_switch_to(top_states.ATTACK_SHOOT)
			is_aiming = false
			return
		top_states.ATTACK_SHOOT:
			if is_attacking:
				return
			is_attacking = true
			
			var target_height_difference: float = self.global_position.y - active_target.global_position.y
			
			if absf(target_height_difference) > 5.0:
				active_weapon_last_raycast_rotation = active_weapon.rotation
				active_weapon.raycast.global_rotation = sight_raycast.global_rotation
				if target_height_difference > 0:
					pass # Under
				elif target_height_difference < 0:
					pass # Over
			else:
				active_weapon.raycast.rotation = active_weapon_last_raycast_rotation
				
				if is_using_weapon_raycast and active_weapon.get("raycast"):
					active_weapon.raycast.enabled = true
					active_weapon.raycast.add_exception(self)
			active_weapon.trigger_down(self)
			
			await get_tree().create_timer(shoot_duration).timeout
			active_weapon.trigger_up()
			await get_tree().create_timer(shoot_cooloff).timeout
			
			if is_dead:
				var dropped_weapon: RigidBody3D = U.spawn_world_weapon(active_weapon.hud_name, rhand_anchor.global_position)
				dropped_weapon.apply_central_impulse(Vector3(0, 5, 0))
				active_weapon.queue_free()
			elif not is_dead:
				top_state_switch_to(top_states.ALERT)
			is_attacking = false
			last_attack_time = time
			return
		top_states.ATTACK_MELEE_INTENT:
			return
		top_states.ATTACK_MELEE:
			return
		top_states.MOVE_DIRECTION:
			#umove(
				#Vector2.UP * (move_speed * umove_conversion_scale)
			#)
			if travel_distance < 0 or not move_direction:
				is_running = false
				umove_expire.stop()
				ticker_interval = BASE_TICKER_INTERVAL
				A.face_position(self, last_seen_target_position)
				top_state_switch_to(top_states.ALERT)
				return
			
			is_running = true
			if umove_expire.is_stopped():
				umove_expire.start()
			
			var applied_move: Vector2
			if is_move_blind:
				applied_move = Vector2.UP * (move_speed * umove_conversion_scale)
			elif not is_move_blind:
				applied_move = move_direction * (move_speed * umove_conversion_scale)
			
			var premove_pos: Vector3 = self.global_position
			umove(applied_move)
			#var travel_loss: float = (premove_pos - self.global_position).abs().length()
			var travel_loss: float = absf(applied_move.length())
			travel_distance -= travel_loss
			return
		top_states.MOVE_DESTINATION:
			return
		top_states.DEAD:
			pass
			return
		top_states.NONE:
			return
	return

func umove(plane_dir: Vector2 = Vector2.ZERO, stop_on_collision: bool = true, stop_bounce: float = 0.0):
	if stop_on_collision and is_collision_detected():
		return
	reset_umarker()
	
	var col_check_dir: Vector2
	if plane_dir.x < 0:
		col_check_dir = Vector2.LEFT
	elif plane_dir.x > 0:
		col_check_dir = Vector2.RIGHT
	elif plane_dir.x == 0 and plane_dir.y < 0:
		col_check_dir = Vector2.UP
	elif plane_dir.x == 0 and plane_dir.y > 0:
		col_check_dir = Vector2.DOWN
	set_collision_check_direction(col_check_dir)
	
	umarker.position.x = plane_dir.x
	umarker.position.z = plane_dir.y
	previous_umove_position = self.global_position
	self.global_position = umarker.global_position
	reset_umarker()
	return
	
func reset_umarker():
	umarker.position = Vector3.ZERO
	umarker.rotation = Vector3.ZERO
	return

func is_collision_detected():
	for ray in collision_ray_container.get_children():
		if ray.is_colliding():
			return true
	return false

func equip_weapon(weapon_name: String):
	if active_weapon:
		prints(self.name, "tried to equip but active_weapon is not null.")
		return
	active_weapon = Weapons.registry[weapon_name]["scene"].instantiate()
	active_weapon.prop_weapon = true
	rhand_anchor.add_child(
		active_weapon
	)
	
	match active_weapon.size:
		U.weapon_sizes.LONG:
			pass
		U.weapon_sizes.SHORT:
			anim_set["idle"] = "armedshort_idle"
			anim_set["run"] = "armedshort_run"
			anim_set["aim"] = "armedshort_idle"
			anim_set["shoot"] = "armedshort_idle"
		U.weapon_sizes.MICRO:
			anim_set["idle"] = "armedmicro_idle"
			anim_set["run"] = "armedmicro_run"
			anim_set["aim"] = "armedmicro_idle"
			anim_set["shoot"] = "armedmicro_idle"
		U.weapon_sizes.PISTOL:
			anim_set["idle"] = "armed_idle"
			anim_set["run"] = "armed_run"
			anim_set["aim"] = "armedpistol_idle"
			anim_set["shoot"] = "armedpistol_idle"
	return

func update_active_target():
	# Get the closest target and set it as the active target
	target_board.clear()
	
	var target_candidates: Array = []
	for group in groups_i_dont_like:
		var group_nodes: Array = get_tree().get_nodes_in_group(group)
		if not group_nodes:
			continue
		target_candidates.append_array(group_nodes)
	
	if not target_candidates:
		return
	
	for candidate in target_candidates:
		var distance: float = self.global_position.distance_to(candidate.global_position)
		if distance > target_distance_threshold:
			continue
		target_board[candidate] = {
			"distance": distance,
			"name": candidate.name,
			"instance_id": candidate.get_instance_id()
		}
	
	if not target_board:
		passive_target = active_target
		active_target = null
		return
	
	var closest_target: Dictionary = {"distance": -1}
	for target in target_board.keys():
		if target.name.contains("noko") and dbg_node_name == "mook_lite_5":
			pass
		var view_check_result: Dictionary = alternate_view_check(target)
		if view_check_result.is_empty():
			continue
		if view_check_result["collider"] != target:
			continue
		if closest_target["distance"] < 1:
			closest_target = target_board[target]
			continue
		elif closest_target["distance"] < target_board[target]["distance"]:
			closest_target = target_board[target]
			continue
		

	
	if closest_target["distance"] < 1:
		return
	
	active_target = instance_from_id(closest_target["instance_id"])
	
	#if logic_priority and is_in_world_queue:
		#world_queue.remove_from_execution_group(main_group_name, self)
	#
	#if not random_remembered_target_position:
		#random_remembered_target_position = active_target.global_position
	#else:
		#var remember_possibility: float = .5
		#if randf_range(0.0, 0.99) < remember_possibility:
			#random_remembered_target_position = active_target.global_position
	return

func animation_control():
	if is_dead:
		if not anim.assigned_animation == anim_set["die"]:
			anim.play(anim_set["die"])
		return
	
	if is_flinching:
		var d: String = anim.assigned_animation
		if anim.assigned_animation != anim_set["flinch"]:
			anim.play(anim_set["flinch"])
		return
	if is_aiming:
		if anim.assigned_animation != anim_set["aim"]:
			anim.play(anim_set["aim"])
		return
	
	if is_attacking:
		if anim.assigned_animation != anim_set["shoot"]:
			anim.play(anim_set["shoot"])
		return
	
	if is_running:
		anim.play(anim_set["run"])
		return
	if not is_running:
		anim.play(anim_set["idle"])
		return
	return

func is_target_attackable():
	var result: bool = active_target and A.is_actor_ahead_approx(self, active_target) and A.is_actor_viewable(self, sight_raycast, active_target)
	if result == true:
		last_seen_target_position = active_target.global_position
	return result

func switch_to_move_state(move_parameters: Dictionary):
	reset_collision_check_direction()
	move_mode = move_parameters["mode"]
	var new_top_state: top_states
	if move_mode == move_modes.DESTINATION:
		move_destination = move_parameters["vector"]
		new_top_state = top_states.MOVE_DESTINATION
	if move_mode == move_modes.DIRECTION:
		move_direction = move_parameters["vector"]
		travel_distance = move_parameters["distance"]
		new_top_state = top_states.MOVE_DIRECTION
	ticker_interval = FAST_TICKER_INTERVAL
	top_state_switch_to(new_top_state)
	return

func get_seconds_since_last_attack() -> float:
	return time - last_attack_time

func set_collision_check_direction(new_direction: Vector2):
	match new_direction:
		Vector2.UP:
			reset_collision_check_direction()
			return
		Vector2.DOWN:
			collision_ray_container.rotation_degrees = Vector3(0, 180, 0)
			return
		Vector2.LEFT:
			collision_ray_container.rotation_degrees = Vector3(0, 90, 0)
			return
		Vector2.RIGHT:
			collision_ray_container.rotation_degrees = Vector3(0, -90, 0)
			return
		
	return

func reset_collision_check_direction():
	collision_ray_container.rotation = Vector3.ZERO
	return

func hearing_control():
	sounds_heard = A.check_for_sounds(self, hearing_distance)
	for sound in sounds_heard:
		var sound_parent: Node3D = sound.get_parent()
		#if sound_parent in target_board:
		if sound_parent.is_in_group("players") and not is_flinching:
			A.face_position(self, sound_parent.global_position)
		#if enable_sound_following:
			#sound_position = sound.global_position
			#is_following_sound = true
			#fatigue.start()
			break
	return

func is_foot_colliding():
	return floor_sensor.is_colliding()

func apply_fake_gravity():
	if not is_foot_colliding():
		self.global_position.y -= gravity_speed
	return

func flinch():
	if is_flinching:
		return
	is_flinching = true
	var flinch_duration: float = .8
	get_tree().create_tween().tween_callback(
		func():
			if not self:
				return
			if self.is_dead:
				return
			is_flinching = false
	).set_delay(flinch_duration)
	return

func alert():
	top_state_switch_to(top_states.ALERT)
	return


func _on_umove_expire_timeout() -> void:
	move_direction = Vector2.ZERO
	return

func alternate_view_check(phys_target: PhysicsBody3D) -> Dictionary:
	if not phys_target:
		return {}
	
	var assumed_world: World = Blackboard.current_world
	var exclude_colliders: Array[RID] = [self.get_rid()]
	var world_raycast_result: Dictionary = U.world_raycast(
		assumed_world,
		sight_raycast.global_position,
		phys_target.global_position,
		exclude_colliders
	)
	#if world_raycast_result.is_empty():
		#return false
	
	#if dbg_node_name == "mook_lite_4":
		#prints(self.name, "alternate_view_check:", world_raycast_result)
	return world_raycast_result
