extends Control

@onready var noko: CharacterBody3D = get_parent()
@export var slot_manager: Node
@onready var left_hand: Node3D
@onready var right_hand: Node3D
@onready var torso: Node3D

@onready var left_hand_label: Label = $left_hand_label
@onready var right_hand_label: Label = $right_hand_label
@onready var health_label: Label = $health_label
@onready var armor_label: Label = $armor_label
@onready var ammo_label: Label = $ammo_label
@onready var selector_label: Label = $selector_label
@onready var chain_status_label: Label = $chain_status_label

@onready var custom_prompt: Control = $custom_prompt
@onready var prompt_label: Label = $custom_prompt/prompt_label
@onready var response_label: Label = $custom_prompt/response_label
var prompt_responses: Array = []

@onready var item_menu: VBoxContainer = $item_menu
var item_menu_base_y_position: float = 413.0
var item_menu_y_offset: float = 50.0
var item_menu_child_height: float = 16.0

var debug_recent_fps_counts: Array = []
var debug_ticker: float = 0.0
var debug_interval: float = 5.0
var debug_gate: bool = true

func _ready():
	noko.connect("ready", _init_parent_vars)
	noko.connect("akimbo_type_changed", _akimbo_type_changed)
	slot_manager.connect("hand_activated", _hand_activated)
	slot_manager.connect("hand_deactivated", _hand_deactivated)
	slot_manager.connect("slot_activated", _slot_activated)
	slot_manager.connect("slot_deactivated", _slot_deactivated)
	slot_manager.connect("ammo_changed", _ammo_changed)
	slot_manager.connect("broadcast_message", _slot_message)
	
	selector_label.text = ""
	
	await get_tree().create_timer(5.0).timeout
	debug_gate = false
	_armor_changed(0) # Set to invisible in a nice way
	return

func _process(delta):
	#$debug_hud.text = "x: " + str(noko.external_carrot_left_base.rotation_degrees.x) + "\ny: " + str(noko.external_carrot_left_base.rotation_degrees.y) + "\nfocus: " + str(noko.akimbo_focus_modes.keys()[noko.akimbo_focus_mode])
	
	var current_fps: float = Engine.get_frames_per_second()
	var debug_append_msg: String = ""
	
	debug_ticker -= delta
	if debug_ticker < 0.0:
		debug_ticker = debug_interval
		# ?
	debug_recent_fps_counts.push_back(current_fps)
	if len(debug_recent_fps_counts) > 20:
		debug_recent_fps_counts.pop_front()
	
	var fps_sum: float = debug_recent_fps_counts.reduce(func(acc, num): return acc + num)
	var fps_average: float = fps_sum / debug_recent_fps_counts.size()
	#print(debug_recent_fps_counts)
	var world_queue: Node3D = get_tree().get_first_node_in_group("world_queue")
	if world_queue:
		$debug_fps_counter.text = "FPS: " + str(current_fps) + "\nAVG: " + str(fps_average) + "\nTicker: " + str(snappedf(debug_ticker, .1)) + "\nQueued Mooks: " + str(len(world_queue.execution_groups["mooks"]))
	else:
		$debug_fps_counter.text = "FPS: " + str(current_fps)
	if not debug_gate and len(debug_recent_fps_counts) > 19 and fps_average < 60.0:
		pass
	
	return

func _slot_message(new_message: String):
	set_doom_message(new_message)
	return

func set_doom_message(new_message: String, duration: float = 0.0):
	$doom_msg.text = new_message
	if duration == 0.0:
		$doom_msg/doom_msg_timer.start()
	else:
		await get_tree().create_timer(duration).timeout
		$doom_msg.text = ""
	return

func set_chain_status(new_text: String, timeout: float = 0.0):
	if new_text == "":
		chain_status_label.visible = false
	else:
		chain_status_label.visible = true
	chain_status_label.text = new_text
	if timeout > 0.00:
		await get_tree().create_timer(timeout).timeout
		chain_status_label.text = ""
	return

func _init_parent_vars():
	right_hand = noko.rhand_anchor
	right_hand.connect("child_entered_tree", _right_hand_equipped)
	left_hand = noko.lhand_anchor
	left_hand.connect("child_entered_tree", _left_hand_equipped)
	torso = noko.torso_anchor
	torso.connect("child_entered_tree", _torso_equipped)
	
	noko.connect("health_changed", _health_changed)
	noko.connect("armor_changed", _armor_changed)
	health_label.text = str(noko.health) + "%"
	return

func _right_hand_equipped(new_child):
	right_hand_label.text = new_child.hud_name
	return

func _left_hand_equipped(new_child):
	left_hand_label.text = new_child.hud_name
	return

func _torso_equipped(new_child):
	left_hand_label.text = ""
	right_hand_label.text = ""
	return


func _health_changed(value: int):
	health_label.text = str(value) + "%"
	return

func _armor_changed(value: int):
	if value < 1:
		armor_label.visible = false
	else:
		armor_label.visible = true
	armor_label.text = str(value) + "%"
	return

func _hand_activated():
	$slot_menu.visible = true
	return

func _hand_deactivated():
	$slot_menu.visible = false
	return

func _slot_activated(slot_num: int, slot_items: Array):
	if len(slot_items) > $item_menu.get_child_count():
		push_error("NOKO HUD: Slot item count exceeds item label count.")
		
	# For each item, duplicate item_ and name it after the item number (item_1, item_2, ...)
	# Slot items I expect are accessed like this:
	#	slot_items[first_gun_in_slot], slot_items[second_gun_in_slot], ...
	# So for each slot_item, follow the index,
	#	make the new UI item,
	#	change text to name
	
	#prints("NOKO HUD: Item list received:", slot_items)
	for item in item_menu.get_children():
		if item.name == "item_":
			continue
		item.queue_free()
	
	var menu_offset: float
	
	var is_only_fist: bool = (len(slot_items) == 1 and slot_items[0]["name"] == "fist")
	if len(slot_items) == 0 or is_only_fist:
		var new_menu_item: ColorRect = item_menu.get_node("item_").duplicate()
		new_menu_item.get_node("item_label").text = "Empty"
		item_menu.add_child(new_menu_item)
		new_menu_item.visible = true
		menu_offset = 0
	else:
		for i in len(slot_items):
			#prints("NOKO HUD: Item", i, slot_items[i])
			#if slot_items[i]["name"].to_lower().contains("fist"):
				#continue
			var displayed_index: int = i + 1
			var new_menu_item: ColorRect = item_menu.get_node("item_").duplicate()
			new_menu_item.name = new_menu_item.name + str(displayed_index)
			if displayed_index <= 4:
				new_menu_item.get_node("item_label").text = str(displayed_index) + " - " + slot_items[i]["name"]
			else:
				new_menu_item.get_node("item_label").text = "    " + slot_items[i]["name"]
			item_menu.add_child(new_menu_item)
			new_menu_item.visible = true
		menu_offset = item_menu_child_height
	
	
	# After the loop, the item_menu size.y should be item_list_height (the length of the item list (minus 1 for item_))
	# And the position.y should be base_y - item_list_height
	var item_list_height: float = item_menu_child_height * len(slot_items)
	item_menu.position.y = (item_menu_base_y_position - item_list_height + menu_offset)
	item_menu.size.y = item_list_height
	
	
	
	
#	match len(slot_items):
#		0:
#			$item_menu/item_1/item_1_label.text = "empty"
#			$item_menu/item_2/item_2_label.text = "empty"
#		1:
#			$item_menu/item_1/item_1_label.text = "1 - " + slot_items[0]["name"]
#			$item_menu/item_2/item_2_label.text = "empty"
#		2:
#			$item_menu/item_1/item_1_label.text = "1 - " + slot_items[0]["name"]
#			$item_menu/item_2/item_2_label.text = "2 - " + slot_items[1]["name"]

	$item_menu.visible = true

	return

func _slot_deactivated():
	$item_menu.visible = false
	return


#func _ammo_changed(hand: int, ammo_count: int):
func _ammo_changed(_hand: int = -1, _ammo_count: int = -1):
	#ammo_label.text = str(hand) + " " + str(ammo_count)
	
	var left_hand_item: Dictionary = noko.slot_manager.get_active_item(U.hands.LEFT)
	var left_mag_ammo: String
	var left_stored_ammo: String
	if not left_hand_item:
		left_mag_ammo = "--"
		left_stored_ammo = "--"
	else:
		left_mag_ammo = str(left_hand_item["ammo_count"]).pad_zeros(2)
		left_stored_ammo = str(noko.slot_manager.ammo_inventory[left_hand_item["ammo_type"]]).pad_zeros(2)

	var right_hand_item: Dictionary = noko.slot_manager.get_active_item(U.hands.RIGHT)
	var right_mag_ammo: String
	var right_stored_ammo: String
	if not right_hand_item:
		right_mag_ammo = "--"
		right_stored_ammo = "--"
	else:
		right_mag_ammo = str(right_hand_item["ammo_count"]).pad_zeros(2)
		right_stored_ammo = str(noko.slot_manager.ammo_inventory[right_hand_item["ammo_type"]]).pad_zeros(2)
	
	var stored_ammo_line: String = left_stored_ammo + "/" + right_stored_ammo
	var mag_ammo_line: String = left_mag_ammo + "/" + right_mag_ammo
	ammo_label.text = stored_ammo_line + "\n" + mag_ammo_line
	return

func _akimbo_type_changed(new_type: int):
	match new_type:
		U.akimbo_types.LINKED:
			selector_label.text = "LINKED"
		U.akimbo_types.UNLINKED:
			selector_label.text = "UNLINKED"
		U.akimbo_types.SWITCH:
			selector_label.text = "SWITCH"
	return

func _on_doom_msg_timer_timeout():
	$doom_msg.text = ""
	return

func hide_hud():
	# Keep doom msg visible
	for child in self.get_children():
		if child.name == "doom_msg":
			continue
		child.visible = false
	return

func update_key(keyname: String):
	for key_icon in $doorkey_display.get_children():
		if keyname.to_lower() == key_icon.name.to_lower():
			key_icon.visible = true
	return

func show_custom_prompt():
	custom_prompt.visible = true
	return

func hide_custom_prompt():
	custom_prompt.visible = false
	return

	#caller.ui_prompt({
		#"responses": ["yes", "no"],
		#"prompt": "Buy ticket?",
		#"callback": "_give_venue_ticket"
	#})
func ui_prompt(prompt_props: Dictionary):
	prompt_responses.clear()
	show_custom_prompt()
	var response_text: String = ""
	var response_index: int = 0
	prompt_responses = prompt_props["responses"]
	for response in prompt_props["responses"]:
		response_index += 1
		response_text += str(response_index) + " " + response + "\n"
	
	prompt_label.text = prompt_props["prompt"]
	response_label.text = response_text
	await noko.ui_prompt_response_submitted
	prompt_props["caller"].call(prompt_props["callback"], noko.ui_prompt_response)
	hide_custom_prompt()
	return

func set_hud_credits(credit_amount: int = -1):
	var displayed_credits: String
	#if credit_amount < 0: # Leaving commented because negative may be a bug hint
		#displayed_credits = "0"
	displayed_credits = str(credit_amount) + "CR"
	$credits_display.text = displayed_credits
	return
