extends Node

signal hand_activated()
signal hand_deactivated()
signal slot_activated(slot_num: int, slot_items: Array[String])
signal slot_deactivated()
signal hand_state_changed(state: int)
signal ammo_changed(hand: int, mag_count: int)
signal broadcast_message(message: String)

@onready var actor: CharacterBody3D = get_parent()
@export var left_hand: Node3D
@export var right_hand: Node3D
@export var torso: Node3D

@onready var hand_nodes: Array[Node3D] = [left_hand, right_hand]
var active_hand: int = U.hands.NONE
var active_slot: int = 0
var active_items: Dictionary = {
	U.hands.LEFT: {},
	U.hands.RIGHT: {}
}

const DEFAULT_ACTIVE_ITEMS: Dictionary = {
	U.hands.LEFT: {},
	U.hands.RIGHT: {}
}


enum slot_steps {OFF, HAND, SLOT, ITEM}
var slot_step: int = slot_steps.HAND
var slot_inventory: Dictionary = {
	0: [], # Dunno what would ever be in 0. 
	1: [{ "name": "fist", "slot_num": 1, "scene": preload("res://Scenes/fist.tscn"), "iid": Weapons.FIST_IID }], # Melee: fists and stuff. Katana?	
	2: [], # Pistols: 9mm, 50, something cool from Twitter
	3: [], # Shotguns: sawedoff, pump
	4: []  # Autos: smg, grunt rifle, something cool from Twitter
}

var ammo_inventory: Dictionary = {
	U.ammo_types.NINE_MM: 0,
	U.ammo_types.FORTY_FIVE: 0,
	U.ammo_types.FIFTY: 0,
	U.ammo_types.NATO: 0,
	U.ammo_types.SEVEN_SIXTY_TWO: 0,
	U.ammo_types.THREE_FIFTY_SEVEN: 0,
	U.ammo_types.SHELLS: 0
}


const DEFAULT_INVENTORY: Dictionary = {
	0: [],
	1: [{ "name": "fist", "slot_num": 1, "scene": preload("res://Scenes/fist.tscn"), "iid": Weapons.FIST_IID }],
	2: [],
	3: [],
	4: []
}

const DEFAULT_AMMO: Dictionary = {
	U.ammo_types.NINE_MM: 0,
	U.ammo_types.FORTY_FIVE: 0,
	U.ammo_types.FIFTY: 0,
	U.ammo_types.NATO: 0,
	U.ammo_types.SEVEN_SIXTY_TWO: 0,
	U.ammo_types.SHELLS: 0
}

var is_any_slot_activated: bool = false
var hand_state: int = U.hand_states.EMPTY
const TORSO_ITEM_OFFSET_Z: float = 3

var last_slot_sequence: Dictionary = {
	U.hands.LEFT: {"slot": 1, "item": 1},
	U.hands.RIGHT: {"slot": 1, "item": 1}
}


func request(number_pressed: int):
	# FLOW
	# [HAND] tap 1-2 > [SLOT] see 4 slots, tap 1-4 > [ITEM] see two slots, tap 1-2 > [HAND] see no slots, equip item
	# 
	match slot_step:
		slot_steps.HAND:
			active_hand = number_pressed
			hand_activated.emit()
			slot_step = slot_steps.SLOT
			return
		
		slot_steps.SLOT:
			hand_deactivated.emit()
			active_slot = number_pressed
			var item_list: Array = slot_inventory[active_slot] # FIXME: Make a list of strings with name and ammo left (e.g. pistol 9mm (3))
			#prints("SLOT MANAGER: Item list emitted:", item_list)
			slot_activated.emit(number_pressed, item_list)
			slot_step = slot_steps.ITEM
			
			last_slot_sequence[active_hand]["slot"] = active_slot
			return
		
		slot_steps.ITEM:
			slot_deactivated.emit()
			var item_idx: int = (number_pressed - 1)
			
			var is_fist_chosen: bool = (number_pressed == 1 and active_slot == 1)
			if is_fist_chosen:
				#equip_fist(active_hand)
				#set_hand_state()
				slot_step = slot_steps.HAND
				
				#last_slot_sequence[active_hand]["item"] = item_idx
				return
			
			#var item_idx: int = (number_pressed - 1)
			var slot_item_count: int = len(slot_inventory[active_slot])
			
			# Added to try and fix the punching big \/
			if slot_item_count == 0:
				slot_step = slot_steps.HAND
				return
			# Added to try and fix the punching big /\
			
			var new_item: Dictionary
			if number_pressed > slot_item_count:
				#prints("Item choice", number_pressed, "is out of range in slot", active_slot, "!")
				#print("--> ", slot_inventory[active_slot])
				new_item = {}
			elif len(slot_inventory[active_slot]) < 1:
				new_item = {}
			else:
				new_item = slot_inventory[active_slot][item_idx]
			
			
			if not new_item.is_empty():
				var hand_idx: int = (active_hand - 1)
				equip_item(hand_nodes[hand_idx], new_item, active_hand)
			
			set_hand_state()
			slot_step = slot_steps.HAND
			
			last_slot_sequence[active_hand]["item"] = item_idx
			return

	return

func equip_unarmed():
	var max_torso_children: int = 5
	if torso.get_child_count() >= max_torso_children:
		#print("Tried to attach fist when torso already has " + str(torso.get_child_count()) + ". Not attaching.")
		return
	
	var fist: Node3D = Weapons.registry["fist"]["scene"].instantiate()
	torso.add_child(fist)
	fist.position.z += TORSO_ITEM_OFFSET_Z
	return



func equip_item(parent_node: Node3D, item: Dictionary, hand: int):
	if item.is_empty():
		prints(self.name, "exited after trying to equip an empty item.")
		return
	
	if item and item["name"] == "fist":
		free_torso()
		if hand == U.hands.LEFT:
			free_left_hand()
		elif hand == U.hands.RIGHT:
			free_right_hand()
		return
	
	if item and item["name"] != "fist":
		actually_free_torso()
		if hand == U.hands.LEFT:
			actually_free_left_hand()
		elif hand == U.hands.RIGHT:
			actually_free_right_hand()
	# FIXME:
	# Need to add logic for melee weapons (like crowbar etc.)
	# FIXME:
	# In certain situations a million fists spawn
	# Maybe just free the torso always?
	# Also sometimes punches do nothing after going unarmed
	
	if item and item["name"] == "fist":
		return
	
	var slot_num: int = item["slot_num"]
	var slot_item_idx: int = -1
	var slot_item_count: int = len(slot_inventory[slot_num])
	for i in range(slot_item_count):
		if slot_inventory[slot_num][i]["iid"] == item["iid"]:
			slot_item_idx = i
			break
	
	prints(self.name, item["iid"])
	
	if slot_item_idx == -1:
		var err_msg: String = "iid "+ str(item["iid"]) +" not found in slot " + str(slot_num)
		push_error(err_msg)
	
	active_items[hand] = slot_inventory[slot_num].pop_at(slot_item_idx)

	var scene: Resource = item["scene"]
	var new_equip: Node3D = scene.instantiate()
	
	new_equip.hand = hand
	if item.has("weapon_theme"):
		new_equip.weapon_theme = item["weapon_theme"]
	
	parent_node.add_child(new_equip)
	new_equip.actor = actor
	
	if new_equip.has_node("RayCast3D"):
		new_equip.get_node("RayCast3D").add_exception(actor)
		for civ in get_tree().get_nodes_in_group("civilians"):
			new_equip.get_node("RayCast3D").add_exception(civ)
	elif new_equip.has_node("raycast_container"):
		for raycast in new_equip.get_node("raycast_container").get_children():
			raycast.add_exception(actor)
			for civ in get_tree().get_nodes_in_group("civilians"):
				raycast.add_exception(civ)
	
	if "ammo_type" in active_items[hand]:
		ammo_changed.emit(hand, active_items[hand]["ammo_count"])
	set_hand_state()
	return

func equip_fist(hand: int):
	var hand_node_index: int = (hand - 1)
	var slot_num: int = 1
	var fist_index: int = 0
	equip_item(hand_nodes[hand_node_index], slot_inventory[slot_num][fist_index], hand)
	return

func equip_item_by_iid(hand: int, iid: int):
	if iid == -1:
		return
	var hand_node_index: int = (hand - 1)
	var parent_node: Node3D = hand_nodes[hand_node_index]
	var item_dict: Dictionary
	
	for slot in slot_inventory.keys():
		if item_dict: break
		for item in slot_inventory[slot]:
			if item["iid"] == iid:
				item_dict = item
				break
	
	equip_item(parent_node, item_dict, hand)
	return

func actually_free_left_hand():
	if not active_items[U.hands.LEFT].is_empty():
		var item: Dictionary = active_items[U.hands.LEFT]
		slot_inventory[item["slot_num"]].append(item)
		active_items[U.hands.LEFT] = {}
	
	for held_item in left_hand.get_children():
		held_item.free()
	return

func actually_free_right_hand():
	if not active_items[U.hands.RIGHT].is_empty():
		var item: Dictionary = active_items[U.hands.RIGHT]
		slot_inventory[item["slot_num"]].append(item)
		active_items[U.hands.RIGHT] = {}
	
	for held_item in right_hand.get_children():
		held_item.free()
	return



func free_left_hand(trigger_hand_state: bool = true):
	if not active_items[U.hands.LEFT].is_empty():
		var item: Dictionary = active_items[U.hands.LEFT]
		slot_inventory[item["slot_num"]].append(item)
		active_items[U.hands.LEFT] = {}
	
	for held_item in left_hand.get_children():
		held_item.free()
	
	if trigger_hand_state:
		set_hand_state()
	return

func free_right_hand(trigger_hand_state: bool = true):
	if not active_items[U.hands.RIGHT].is_empty():
		var item: Dictionary = active_items[U.hands.RIGHT]
		slot_inventory[item["slot_num"]].append(item)
		active_items[U.hands.RIGHT] = {}
	
	for held_item in right_hand.get_children():
		held_item.free()
	
	if trigger_hand_state:
		set_hand_state()
	return

func drop_left_hand():
	if not active_items[U.hands.LEFT].is_empty():
		var item: Dictionary = active_items[U.hands.LEFT]
		Global.remove_iid(item["iid"])
		active_items[U.hands.LEFT] = {}
	
	for held_item in left_hand.get_children():
		held_item.free()
	
	set_hand_state()
	return

func drop_right_hand():
	if not active_items[U.hands.RIGHT].is_empty():
		var item: Dictionary = active_items[U.hands.RIGHT]
		Global.remove_iid(item["iid"])
		active_items[U.hands.RIGHT] = {}
	
	for held_item in right_hand.get_children():
		held_item.free()
	
	set_hand_state()
	return

func free_torso():
	var torso_children: Array = torso.get_children()
	#for held_item in torso.get_children():
	for held_item in torso_children:
		#held_item.queue_free()
		held_item.free()
	return

func actually_free_torso():
	free_torso() # Probably not broken!
	return

func equip_both_fists():
	free_left_hand()
	free_right_hand()
	free_torso()
	equip_unarmed()
	return

func primary_action():
	var action_event: int = -1
	
	match hand_state:
		U.hand_states.EMPTY:
			# If empty and torso has child
			# - Melee attack (e.g. punch)
			# If empty and torso has no child
			# - I don't know, throw an error. This probably shouldn't happen.
			
			if not actor.is_punch_allowed:
				return U.action_events.IGNORED
			
			melee_attack()
			if hand_state == U.hand_states.EMPTY:
				action_event = U.action_events.MELEE_PUNCH
				return action_event
			else:
				action_event = U.action_events.MELEE_HELD
				return action_event
		
		# For singles, the shooting type will be determined by the weapon.
		# For example the pistol_9mm will always TAP and smgs will always HOLD.
		U.hand_states.SINGLE_LEFT:
			action_event = U.action_events.SHOOT
			return action_event
		U.hand_states.SINGLE_RIGHT:
			action_event = U.action_events.SHOOT
			return action_event
		
		# For akimbo there are two modes:
		# - Linked
		# 		Primary action executes on both.
		# 		A HOLD weapon gets a HOLD action
		#		A TAP weapon gets a TAP action at predetermined intervals (same for all for now)
		# - Unlinked
		#		The focus button is now the primary action key for the left hand.
		U.hand_states.AKIMBO:
			action_event = U.action_events.SHOOT
			return action_event

	return


func mod_action():
	var action_event: int = -1
	
	match hand_state:
		U.hand_states.EMPTY:
			if not actor.is_punch_allowed:
				return U.action_events.IGNORED
			
			mod_melee_attack()
			if hand_state == U.hand_states.EMPTY:
				action_event = U.action_events.MELEE_PUNCH
				return action_event
			else:
				action_event = U.action_events.MELEE_HELD
				return action_event
		
		U.hand_states.SINGLE_LEFT:
			action_event = U.action_events.SHOOT
			return action_event
		U.hand_states.SINGLE_RIGHT:
			action_event = U.action_events.SHOOT
			return action_event
		
		U.hand_states.AKIMBO:
			action_event = U.action_events.SHOOT
			return action_event
	return

func melee_attack():
	if torso.get_child_count() == 1:
		var torso_first_child: Node3D = torso.get_child(0)
		torso.get_child(0).primary_action(actor, U.primary_action_types.TAP)
	if torso.get_child_count() == 0:
		push_error(self.name + " Torso has no children and melee_attack called. Bad!")
	# torso has no child?
	# - I don't know, throw an error. This probably shouldn't happen.
	return

func mod_melee_attack():
	if torso.get_child_count() == 1:
		var torso_first_child: Node3D = torso.get_child(0)
		torso.get_child(0).mod_action(actor)
	if torso.get_child_count() == 0:
		push_error(self.name + " Torso has no children and mod_melee_attack called. Bad!")
	return

func get_active_item(hand: int):
	return active_items[hand]

func get_held_weapon(hand: int) -> Node3D:
	var hand_node_idx: int = (hand - 1)
	var hand_node: Node3D = hand_nodes[hand_node_idx]
	if hand_node.get_child_count() == 0:
		return
	
	var weapon: Node3D = hand_node.get_children()[0]
	return weapon

func suppress_weapon_dots():
	for hand in hand_nodes:
		if len(hand.get_children()) < 1:
			continue
		var assumed_weapon: Node3D = hand.get_children()[0]
		assumed_weapon.suppress_aimdot = true
	return

func reset_weapon_dots():
	for hand in hand_nodes:
		if len(hand.get_children()) < 1:
			continue
		var assumed_weapon: Node3D = hand.get_children()[0]
		assumed_weapon.suppress_aimdot = false
	return

func weapon_attack(hand: int):
	var hand_node_idx: int = (hand - 1)
	var hand_node: Node3D = hand_nodes[hand_node_idx]
	if hand_node.get_child_count() == 0:
		return
	
	var weapon: Node3D = hand_node.get_children()[0]
	if not "ammo_type" in active_items[hand]:
		prints("Not an ammo item:", active_items[hand])
		weapon.primary_action(actor)
	else:
		#prints("Is an ammo item:", active_items[hand])

		# If there are 1 or more bullets, attack
		if active_items[hand]["ammo_count"] > 0:
			weapon.primary_action(actor)
			if weapon.primary_action_type == U.primary_action_types.TAP:
				active_items[hand]["ammo_count"] -= 1
			elif weapon.primary_action_type == U.primary_action_types.HOLD:
				# FIXME: Subtraction should match fire rate. This is a dirty hack?
				#active_items[hand]["ammo_count"] -= 1
				weapon.callback_data = {"hand": hand, "iid": active_items[hand]["iid"]}
				weapon.finished_shot_callback = self._weapon_finished_shot
			
			var is_empty_after_shot: bool = active_items[hand]["ammo_count"] == 0
			if is_empty_after_shot:
				weapon.show_as_empty()
			ammo_changed.emit(hand, active_items[hand]["ammo_count"])
		else:
			weapon.dry_fire(actor)
	return

func weapon_cease(hand: int):
	var hand_node_idx: int = (hand - 1)
	var hand_node: Node3D = hand_nodes[hand_node_idx]
	if hand_node.get_child_count() == 0:
		return
	
	var weapon: Node3D = hand_node.get_children()[0]
	weapon.stop_action()
	return

func weapon_reload(hand: int):
	var hand_node_idx: int = (hand - 1)
	var hand_node: Node3D = hand_nodes[hand_node_idx]
	if hand_node.get_child_count() == 0:
		return
	
	var weapon: Node3D = hand_node.get_children()[0]
	weapon.stop_action() # NEW, try to fix the forever shooting thing with full autos after reload
	
	var is_reload_successful: bool
	if weapon.fire_mode == U.fire_modes.SINGLE_ACTION:
		is_reload_successful = load_cartridge(active_items[hand])
	else:
		is_reload_successful = fill_mag(active_items[hand])
	if is_reload_successful:
		weapon.reload(actor)
		ammo_changed.emit(hand, active_items[hand]["ammo_count"])
	return is_reload_successful

func add_ammo(ammo_type: int, ammo_count: int):
	ammo_inventory[ammo_type] += ammo_count
	return

func remove_ammo(ammo_type: int, ammo_count: int):
	ammo_inventory[ammo_type] -= ammo_count
	return

func add_item(item_name: String, weapon_theme: int = -1):
	var new_item: Dictionary = Weapons.registry[item_name].duplicate()
	var slot: int = new_item["slot_num"]
	
	if weapon_theme > -1:
		new_item["weapon_theme"] = weapon_theme
	
	var same_item_count: int = 0
	for slot_item in slot_inventory[slot]:
		if slot_item["name"] == item_name:
			same_item_count += 1
	
	for hand in [U.hands.LEFT, U.hands.RIGHT]:
		if not active_items[hand]:
			continue
		if active_items[hand]["name"] == item_name:
			same_item_count += 1
	
	if same_item_count < 2:
		new_item["iid"] = Global.new_iid(item_name)
		fill_mag(new_item)
		slot_inventory[slot].append(new_item)

	broadcast_message.emit("You picked up " + item_name)
	return {"slot": slot, "item": new_item}

func fill_mag(item: Dictionary):
	if not "ammo_type" in item:
		prints(item["name"], item["iid"], "Tried to fill magazine but no ammo type.")
		return false
	if ammo_inventory[item["ammo_type"]] < 1:
		prints(item["name"], item["iid"], "Tried to fill magazine but <1 ammo of type", item["ammo_type"], "is available.")
		return false
	if item["ammo_count"] == item["mag_size"]:
		prints(item["name"], item["iid"], "Tried to fill magazine but ammo count equals mag size.")
		return false

	var fill_size: int = item["mag_size"] - item["ammo_count"]
	var available_inventory_ammo: int = ammo_inventory[item["ammo_type"]]
	if available_inventory_ammo < fill_size:
		fill_size = available_inventory_ammo

	ammo_inventory[item["ammo_type"]] -= fill_size
	item["ammo_count"] += fill_size
	return true

func load_cartridge(item: Dictionary):
	if not "ammo_type" in item:
		prints(item["name"], item["iid"], "Tried to load cartridge but no ammo type.")
		return false
	if ammo_inventory[item["ammo_type"]] < 1:
		prints(item["name"], item["iid"], "Tried to load cartridge but <1 ammo of type", item["ammo_type"], "is available.")
		return false
	if item["ammo_count"] == item["mag_size"]:
		prints(item["name"], item["iid"], "Tried to load cartridge but ammo count equals mag size.")
		return false
	
	ammo_inventory[item["ammo_type"]] -= 1
	item["ammo_count"] += 1
	return true

func set_hand_state():
	var is_left_empty = (left_hand.get_child_count() == 0)
	var is_right_empty = (right_hand.get_child_count() == 0)
	if is_left_empty and is_right_empty:
		hand_state = U.hand_states.EMPTY
	elif not is_left_empty and is_right_empty:
		hand_state = U.hand_states.SINGLE_LEFT
	elif is_left_empty and not is_right_empty:
		hand_state = U.hand_states.SINGLE_RIGHT
	elif not is_left_empty and not is_right_empty:
		hand_state = U.hand_states.AKIMBO
	else:
		push_error(self.name + " I don't know what happened :( follow the stack trace.")
	
	hand_state_changed.emit(hand_state)
	return

func _weapon_finished_shot(hand: int, iid: int):
	if active_items[hand]["iid"] != iid:
		#print("Weapon finished shot but with the wrong iid.")
		#prints("--> finished iid:", iid, "active iid:", active_items[hand]["iid"])
		return
	
	var shot_count: int = 1
	if active_items[hand].has("consecutive_shots"):
		var max_consecutive_shots: int = active_items[hand]["consecutive_shots"]
		if active_items[hand]["ammo_count"] < max_consecutive_shots:
			shot_count = active_items[hand]["ammo_count"]
		else:
			shot_count = max_consecutive_shots
	active_items[hand]["ammo_count"] -= shot_count
	ammo_changed.emit(hand, active_items[hand]["ammo_count"])
	
	# New, bad big noticed when switch shooting autopistols. The one in Noko's left hand just kept shooting!
	if active_items[hand]["ammo_count"] < 1:
		weapon_cease(hand)
	# This appears to have fixed it but my linked_semi_whatever_ticker probably caused the issue.
	return



func save_active_iids_to_cache():
	var active_item_iids: Dictionary = {}
	for hand_e in [U.hands.LEFT, U.hands.RIGHT]:
		if not active_items[hand_e].is_empty():
			#Global.prop_table["slot_manager_cache"]["held_item_iids"][hand_e] = active_items[hand_e]["iid"]
			active_item_iids[hand_e] = active_items[hand_e]["iid"]
			
	if not active_item_iids.is_empty():
		Global.prop_table["slot_manager_cache"].merge({"active_item_iids": active_item_iids}, true)
	return


func reset_inventory():
	slot_inventory = DEFAULT_INVENTORY.duplicate(true)
	ammo_inventory = DEFAULT_AMMO.duplicate(true)
	active_items = DEFAULT_ACTIVE_ITEMS.duplicate(true)
	return

func save_inventory_to_cache():
	var new_cache: Dictionary = {
		"slot_inventory": slot_inventory,
		"ammo_inventory": ammo_inventory
	}
	if not Global.prop_table.has("slot_manager_cache"):
		Global.prop_table["slot_manager_cache"] = new_cache
	else:
		Global.prop_table["slot_manager_cache"].merge(new_cache, true)
	return

func load_inventory_from_cache():
	slot_inventory = Global.prop_table["slot_manager_cache"]["slot_inventory"]
	ammo_inventory = Global.prop_table["slot_manager_cache"]["ammo_inventory"]
	if Global.prop_table["slot_manager_cache"].has("active_item_iids"):
		var iids: Dictionary = Global.prop_table["slot_manager_cache"]["active_item_iids"]
		if not iids.is_empty():
			for hand_e in iids.keys():
				equip_item_by_iid(hand_e, iids[hand_e])
	return
