extends RigidBody3D

@export var is_headlights_allowed: bool = true
@export var spawn_with_headlights: bool = false

var weight_scale: float = 12.0
const BASE_ENGINE_FORCE: float = 650.0
var engine_force: float = 650.0 * weight_scale
const engine_mod_scale: float = .6
var max_accel: float = 300.00
const steer_mod_scale: float = .6
var steer_amount: float = 0.0
var steer_speed: float = 1.5 / weight_scale # 2.0 / weight_scale
var accel_steer_threshold: float = 0.3
var max_steer_amount: float = 3.0
var min_steer_ratio: float = .2
var roll_amount: float = 0.0
var roll_speed: float = 0.001
var roll_dir: float = 0.0
var max_roll: float = 0.03
var max_pitch_downward: float = 5.0
var max_pitch_upward: float = 15.0
var min_body_height: float = -0.11
var max_body_height: float = 0.33

var vehicle_colliders: Array = []
var is_steering: bool = false
var is_accelerate_pressed: bool = false
var is_reverse_pressed: bool = false
var is_modifier_pressed: bool = false
var steer_dir: float = 0.0
var gear_dir: float = 0.0

var air_time: float = 0.0
var is_landing: bool = false

@onready var chassis_meshes: Array = [
	$ggbot_crown_edit/car_body,
	$ggbot_crown_edit/car_door_left,
	$ggbot_crown_edit/seats,
	$ggbot_crown_edit/steering_control,
	$ggbot_crown_edit/steering_mount,
	$ggbot_crown_edit/car_door_right
]
var chassis_mesh_props: Dictionary = {}
@onready var wheel_front_left: MeshInstance3D = $ggbot_crown_edit/wheel_front_left
@onready var wheel_back_left: MeshInstance3D = $ggbot_crown_edit/wheel_back_left
@onready var wheel_front_right: MeshInstance3D = $ggbot_crown_edit/wheel_front_right
@onready var wheel_back_right: MeshInstance3D = $ggbot_crown_edit/wheel_back_right
@onready var carrot: Marker3D = $carrot_base/carrot

var original_wheel_rotations: Dictionary = {}
var tween_pool: Array = []

@onready var active_cameras: Array = [
	$cameras/mounted_camera_2,
	$cameras/mounted_camera_1
]
var active_camera_index: int = 0
@export var external_camera: Camera3D

@onready var sound_container: Node3D = $Sounds
var is_input_allowed: bool = false
var driver_parent: Node3D
var driver: Node3D
var driver_rotation: Vector3
var is_claim_allowed: bool = true

func _ready() -> void:
	for c in self.find_children("*", "StaticBody3D"):
		self.add_collision_exception_with(c)
		vehicle_colliders.append(c)
	
	vehicle_colliders.append(self)
	
	for c in $sensors.get_children():
		for collider in vehicle_colliders:
			c.add_exception(collider)
	
	for wheel in [wheel_front_left, wheel_back_left, wheel_front_right, wheel_back_right]:
		original_wheel_rotations[wheel] = wheel.rotation_degrees
	
	#if not active_cameras.has(get_viewport().get_camera_3d()):
		#active_cameras.insert(0, get_viewport().get_camera_3d())
	if external_camera:
		active_cameras.insert(0, external_camera)
	
	for mesh in chassis_meshes:
		chassis_mesh_props[mesh] = {
			"base_pos": mesh.position,
			"base_rot": mesh.rotation_degrees
		}
	
	if spawn_with_headlights and is_headlights_allowed:
		$lights/headlights.visible = true
	if not is_headlights_allowed or not spawn_with_headlights:
		$lights/headlights.visible = false
	
	if Blackboard.get("current_world"):
		$editor_only.queue_free()
	return


func _physics_process(delta: float) -> void:
	$Control/values/accel.text = str(absf(self.linear_velocity.z))
	$Control/values/tire_col.text = str(is_tires_colliding())
	$Control/values/steer_amount.text = str(steer_amount)
	$Control/values/roll_amount.text = str(roll_amount)
	###
	
	if is_input_allowed:
		input_control()
	sound_control()
	
	if is_reversing():
		$lights/taillights.visible = true
	else:
		$lights/taillights.visible = false
	if is_tires_colliding() and not self.linear_velocity.is_zero_approx() and not is_accelerating():
		self.linear_velocity = self.linear_velocity.slerp(Vector3.ZERO, .01)
		#self.apply_central_force(self.linear_velocity.slerp(Vector3.ZERO, .2))
	
	safety_check()
	
	apply_steering()
	if is_tires_colliding():
		cosmetic_turn_wheels()
		#cosmetic_chassis_transform()
		#apply_steering()
		roll_amount = 0.0
		if air_time > .5:
			is_landing = true
			get_tree().create_tween().tween_callback(func (): is_landing = false).set_delay(.6)
		air_time = 0.0
	if not is_tires_colliding():
		apply_roll()
		air_time += delta
	return

func _integrate_forces(state: PhysicsDirectBodyState3D) -> void:
	$Control/values/linear_velo.text = str(state.linear_velocity)
	$Control/values/glob_rot.text = str(self.global_rotation_degrees)
	
	if is_tires_colliding():
		shrink_drift(state)
	
	return

func is_tires_colliding():
	for raycast in $sensors.get_children():
		if raycast.is_colliding():
			return true
	return false

func input_control():
	
	if Input.is_action_just_pressed("claim") and is_claim_allowed:
		eject_driver()
		return
	
	if Input.is_action_just_pressed("accessory") and is_headlights_allowed:
		$lights/headlights.visible = !$lights/headlights.visible
		$Sounds/accessory.play()
	
	if Input.is_action_just_pressed("view"):
		active_camera_index = U.cycled_array_index(active_cameras, active_camera_index)
		active_cameras[active_camera_index].make_current()
	
	if is_tires_colliding():
		input_on_ground()
	
	
	if Input.is_action_pressed("strafe_left") and Input.is_action_pressed("mod"):
		roll_intent(Vector2.LEFT)
	if Input.is_action_pressed("strafe_right") and Input.is_action_pressed("mod"):
		roll_intent(Vector2.RIGHT)
	if not Input.is_action_pressed("mod") and not (Input.is_action_pressed("strafe_left") or Input.is_action_pressed("strafe_right")):
		roll_dir = 0.0
	
	if Input.is_action_pressed("mode_shift"):
		is_modifier_pressed = true
	elif not Input.is_action_pressed("mode_shift"):
		is_modifier_pressed = false
	return

func input_on_ground():
	if Input.is_action_pressed("forward"):
		if absf(self.linear_velocity.z) <= max_accel:
			var applied_force: float = engine_force
			if is_modifier_pressed:
				applied_force *= engine_mod_scale
			self.apply_central_force(-self.basis.z * applied_force)
		is_accelerate_pressed = true
	else:
		is_accelerate_pressed = false
	
	if Input.is_action_pressed("backward"):
		if absf(self.linear_velocity.z) <= (max_accel * .80):
			var applied_force: float = engine_force
			if is_modifier_pressed:
				applied_force *= engine_mod_scale
			self.apply_central_force(self.basis.z * applied_force)
		is_reverse_pressed = true
	else:
		is_reverse_pressed = false
	
	var steer_mult: float = 1.0 if not is_reversing() else -3.0
	
	if Input.is_action_pressed("strafe_left") and not Input.is_action_pressed("mod"):
		if is_reverse_pressed:
			steer_intent(Vector2.RIGHT)
		else:
			steer_intent(Vector2.LEFT)
	if Input.is_action_pressed("strafe_right") and not Input.is_action_pressed("mod"):
		if is_reverse_pressed:
			steer_intent(Vector2.LEFT)
		else:
			steer_intent(Vector2.RIGHT)
	
	if Input.is_action_pressed("strafe_left") or Input.is_action_pressed("strafe_right"):
		is_steering = true
	else:
		is_steering = false
	
	
	return

func shrink_drift(physics_state: PhysicsDirectBodyState3D):
	if is_steering:
		return
	var drift_threshold: float = 0.02 # 0.02
	var shrink_amount: float = .8
	if physics_state.angular_velocity.y > drift_threshold:
		physics_state.angular_velocity.y = lerpf(
			physics_state.angular_velocity.y,
			0.0,
			shrink_amount
		)
	return

func cosmetic_turn_wheels():
	for wheel in [wheel_front_right, wheel_front_left, wheel_back_right, wheel_back_left]:
		if wheel.name.contains("front"):
			var turn_amount: float = steer_amount * 77
			if is_reverse_pressed:
				turn_amount *= -1
			wheel.rotation_degrees.y = clampf(turn_amount, -45.0, 45.0)
		
		var wheel_velo: Vector3 = self.linear_velocity
		var wheel_velo_scale: float = .2
		if is_reversing():
			wheel.rotation_degrees.x -= wheel_velo.length() * wheel_velo_scale
		else:
			wheel.rotation_degrees.x += wheel_velo.length() * wheel_velo_scale
	return

func cosmetic_chassis_transform():
	if not is_in_motion():
		return
	
	var accel_ratio: float = (acceleration() / max_accel)
	var steer_ratio: float = (steer_amount / max_steer_amount)
	var append_rotation: Vector3 = Vector3.ZERO
	append_rotation.z += lerpf(0.0, 15.0 * steer_dir, steer_ratio)
	var nose_pitch: float = max_pitch_downward if is_reverse_pressed else max_pitch_upward
	append_rotation.x += lerpf(0.0, nose_pitch, accel_ratio)
	for mesh in chassis_meshes:
		var new_rotation: Vector3 = chassis_mesh_props[mesh]["base_rot"] * Vector3(1, 0, 1)
		new_rotation += append_rotation
		mesh.rotation_degrees = new_rotation
	
	return

func is_reversing():
	return (self.linear_velocity.z > 0 and is_reverse_pressed)

func is_accelerating():
	return (self.linear_velocity.z < 0 and is_accelerate_pressed)

func acceleration():
	return self.linear_velocity.z

func is_in_motion():
	return absf(self.linear_velocity.z) > 0.01

func roll_intent(roll_vector: Vector2):
	roll_vector *= -1
	roll_dir = roll_vector.x
	roll_amount += roll_speed * roll_dir
	return

func apply_roll():
	if not absf(roll_amount) > 0:
		return
	
	#roll_amount = clampf(roll_amount, -max_roll, max_roll)
	#self.global_position.y += absf(self.global_rotation.z) * 0.1
	self.global_rotation.z += roll_amount
	roll_amount -= roll_amount * .01
	return

func steer_intent(steer_vector: Vector2):
	steer_dir = steer_vector.x * -1
	#self.apply_torque(Vector3(0, -steering_force * steer_mult, 0))
	#if absf(self.linear_velocity.z) > accel_steer_threshold:
	var applied_steer: float = steer_speed * steer_dir
	steer_amount += applied_steer
	return

func apply_steering():
	if not absf(steer_amount):
		return
	
	if is_in_motion():
		var applied_steering: float = clampf(steer_amount, -max_steer_amount, max_steer_amount)
		if is_modifier_pressed:
			applied_steering *= steer_mod_scale
		self.rotate_y(deg_to_rad(applied_steering))
	
	var steer_loss_threshold: float = 0.0
	var steer_loss: float = steer_amount * .02 # maybe scale it by the ratio to max steer?
	if is_modifier_pressed:
		steer_loss *= 4.0
	if absf(steer_amount) > steer_loss_threshold:
		#steer_amount -= steer_amount * .02 # Works
		steer_amount -= steer_loss
	return

func sound_control():
	#if absf(self.linear_velocity.z) > .0:
	if is_landing and not $Sounds/drop_land.playing:
		$Sounds/drop_land.play()
	
	if is_accelerate_pressed or is_reverse_pressed:
		#solo_sound("engine_run")
		if not $Sounds/engine_run.playing:
			$Sounds/engine_run.play()
			$Sounds/engine_idle.stop()
		
		var accel_ratio: float = (absf(acceleration()) / max_accel)
		$Sounds/engine_run.pitch_scale = lerpf(.80, 1.13, accel_ratio)
	else:
		#$Sounds/engine_run.stop()
		#$Sounds/engine_idle.playing = true
		if not $Sounds/engine_run.playing:
			$Sounds/engine_run.play()
			$Sounds/engine_idle.stop()
		
		$Sounds/engine_run.pitch_scale = .41
	
	if absf(steer_amount) > 3.0 and not $Sounds/tire_screech.playing:
		$Sounds/tire_screech.play()
	elif absf(steer_amount) <= 3.0 and $Sounds/tire_screech.playing:
		$Sounds/tire_screech.stop()
	return

func solo_sound(sound_name: String):
	for sound in sound_container.get_children():
		var sound_node_name: String = sound.name
		if sound.name == sound_name and not sound.playing:
			sound.play()
			continue
		if sound.name != sound_name:
			sound.stop()
	return


func _on_hurt_zone_body_entered(body: Node3D) -> void:
	if is_zero_approx(acceleration()):
		return
	if body.is_in_group("hit_takers"):
		var damage: int = int(absf(acceleration()) * 2)
		if (body.health - damage) < 1 and body.get("force_phys_death"):
			body.force_phys_death = true
		body.hit(damage, U.hit_types.UNDEFINED, null, self.global_position)
		$Sounds/body_hit.play()
		if is_instance_of(body, CharacterBody3D):
			body.velocity.y += 25.0
		
		if body.is_dead:
			self.add_collision_exception_with(body)
	return

func claimed_by(new_driver: Node3D):
	if not is_claim_allowed:
		return
	is_claim_allowed = false
	if driver:
		prints(self.name, "Already a driver, claim aborted.")
		return
	
	driver = new_driver
	driver_rotation = driver.global_rotation
	driver_parent = driver.get_parent()
	self.add_collision_exception_with(driver)
	driver.mount_vehicle(self)
	driver.global_position = $seat_positions/driver_seat.global_position
	#driver.global_rotation = $seat_positions/driver_seat.global_rotation * Vector3(1, 0, 1)
	driver.global_rotation = self.global_rotation
	driver.reparent(self)
	is_input_allowed = true
	await get_tree().create_timer(2.0).timeout
	is_claim_allowed = true
	return

func activate_camera():
	active_cameras[active_camera_index].make_current()
	return

func eject_driver():
	driver.dismount_vehicle()
	driver.global_position = $dismount_positions/driver_seat.global_position
	driver.velocity.y += 10.0
	driver.global_rotation = driver_rotation
	is_input_allowed = false
	
	driver.reparent(driver_parent)
	self.remove_collision_exception_with(driver)
	driver = null
	driver_parent = null
	return

func safety_check():
	# Might not work the way you think it will!
	if not driver:
		$hurt_zone.process_mode = Node.PROCESS_MODE_DISABLED
	else:
		$hurt_zone.process_mode = Node.PROCESS_MODE_INHERIT
	return
