extends Node3D

signal execution_ticker_expired(execution_node: Node)

var execution_groups: Dictionary = {}
var execution_tickers: Dictionary = {}
@export var execution_delay: float = 0.1 # 0.1
@export var strict_execution_delay: float = 0.05
var interval_delta: float = 0.0
var interval_group_index: int = 0

enum execution_modes {
	DISABLED,
	SERIAL,
	EXPIRATION,
	INTERVAL,
	INTERVAL_STRICT
}

@export var include_execution_groups: Array[String] = []
@export var shuffle_execution_groups: bool = false
@export var execution_mode: execution_modes

func _ready():
	for group in include_execution_groups:
		execution_groups[group] = get_tree().get_nodes_in_group(group)
		for group_node in execution_groups[group]:
			if group_node.process_mode == PROCESS_MODE_DISABLED:
				continue
			if group_node.get("reject_world_queue"):
				remove_from_execution_group(group, group_node)
				continue
			group_node.is_in_world_queue = true
			group_node.world_queue = self
			if execution_mode == execution_modes.INTERVAL:
				group_node.soft_process_disable = true
			execution_tickers[group_node] = execution_delay
		
		if shuffle_execution_groups:
			execution_groups[group].shuffle()
	
	self.connect("execution_ticker_expired", _execution_ticker_expired)
	
	if execution_mode == execution_modes.INTERVAL_STRICT:
		execution_delay = strict_execution_delay
	return

func _physics_process(delta):
	match execution_mode:
		execution_modes.DISABLED:
			pass
		execution_modes.SERIAL:
			for group in execution_groups.keys():
				if len(execution_groups[group]) == 0:
					continue
				
				var group_queue: Array = execution_groups[group]
				var elected_node: Node3D = group_queue.pop_front()
				if elected_node.get("reject_world_queue"): # Why wasn't this already caught?
					self.remove_from_execution_group(group, elected_node)
					continue
				elected_node._world_queue_execute()
				group_queue.push_back(elected_node)
		execution_modes.EXPIRATION:
			for target_node in execution_tickers.keys():
				execution_tickers[target_node] -= delta
				if execution_tickers[target_node] < 0:
					execution_tickers[target_node] = execution_delay
					execution_ticker_expired.emit(target_node)
		execution_modes.INTERVAL:
			interval_delta -= delta
			if interval_delta < 0:
				var next_executor: Node = get_next_executor()
				next_executor.soft_process_disable = false
				var process_expire_time: float = 1.0
				get_tree().create_tween().tween_callback(
					func(): next_executor.soft_process_disable = true
				).set_delay(process_expire_time)
				interval_delta = execution_delay
				execute_next_in_queue()
		execution_modes.INTERVAL_STRICT:
			interval_delta -= delta
			if interval_delta < 0:
				var next_executor: Node = get_next_executor()
				if not next_executor:
					return
				var process_expire_time: float = 1.0
				
				next_executor.soft_process_disable = false
				get_tree().create_tween().tween_callback(
					func(): next_executor.soft_process_disable = true
				).set_delay(process_expire_time)
				
				next_executor.process_mode = PROCESS_MODE_DISABLED
				get_tree().create_tween().tween_callback(
					func(): next_executor.process_mode = PROCESS_MODE_INHERIT
				).set_delay(process_expire_time)
				
				interval_delta = execution_delay
				execute_next_in_queue()
	return

func remove_from_execution_group(group_name: String, node_to_remove: Node3D):
	if not node_to_remove.get("reject_world_queue"):
		node_to_remove.is_in_world_queue = false
	execution_groups[group_name].erase(node_to_remove)
	return

func tick_delay_control(group: String):
	return

func _execution_ticker_expired(execution_node: Node):
	execution_node._world_queue_execute()
	return

func execute_next_in_queue():
	var group: String = execution_groups.keys()[interval_group_index]
	interval_group_index += 1
	if interval_group_index > (len(execution_groups.keys()) - 1):
		interval_group_index = 0
	
	var group_queue: Array = execution_groups[group]
	var elected_node: Node3D = group_queue.pop_front()
	elected_node._world_queue_execute()
	group_queue.push_back(elected_node)
	return

func get_next_executor():
	var group: String = execution_groups.keys()[interval_group_index]
	var group_queue: Array = execution_groups[group]
	return group_queue[0]
