extends CharacterBody3D


const SPEED = 14.0
const JUMP_VELOCITY = (4.5 * 5)

# Get the gravity from the project settings to be synced with RigidBody nodes.
var gravity = (ProjectSettings.get_setting("physics/3d/default_gravity") * 6)
@onready var cam_base: Node3D = $cam_base
@onready var graphics: Node3D = $graphics
@onready var chest_raycast: RayCast3D = $chest_raycast
@onready var rhand_anchor: Node3D = $graphics/noko/Armature/Skeleton3D/attach_rhand/anchor
@onready var lhand_anchor: Node3D = $graphics/noko/Armature/Skeleton3D/attach_lhand/anchor
@onready var noko_anim: AnimationPlayer = $graphics/noko/AnimationPlayer
@onready var special_timer: Timer = $special_timer
@onready var hitbox_lower_anim: AnimationPlayer = $hitbox_lower/AnimationPlayer
var mouse_sensitivity: float = 0.002
var run_scale: float = 2
var crouch_scale: float = 0.5
var move_slide_scale: float = 0.5
var is_crouching: bool = false
var is_jumping: bool = false
var is_kicking: bool = false
var crouch_cam_offset_y: float = 3

enum armed_types {
	UNARMED,
	ARMED,
	AKIMBO
}
var armed_type: int = armed_types.UNARMED


var primary_action_type: int = U.primary_action_types.HOLD

var queued_anims: Array[String] = []
var last_anim: String = ""
var anim_set: Dictionary = {
	"idle": "unarmed_idle",
	"run": "unarmed_run",
	"jump": "unarmed_jump",
	"crouch": "crouch"
}

func _ready():
	Input.mouse_mode = Input.MOUSE_MODE_CAPTURED
	anim_play(anim_set["idle"])
	return

func _unhandled_input(event):
	if event is InputEventMouseMotion:
		cam_base.rotation.x -= (event.relative.y * mouse_sensitivity)
		cam_base.rotation_degrees.x = clamp(cam_base.rotation_degrees.x, -90, 90)
		#cam_base.rotation.y -= (event.relative.x * mouse_sensitivity)
		graphics.rotation.x -= (event.relative.y * mouse_sensitivity)
		graphics.rotation_degrees.x = clamp(graphics.rotation_degrees.x, -90, 90)
		self.rotation.y -= (event.relative.x * mouse_sensitivity)
	
	return

func _physics_process(delta):
	# Add the gravity.
	if not is_on_floor():
		velocity.y -= gravity * delta

	# Handle Jump.
	if Input.is_action_just_pressed("jump") and (is_on_floor() or chest_raycast.is_colliding()):
		velocity.y = JUMP_VELOCITY
		anim_play(anim_set["jump"], false)
		is_jumping = true

	# Get the input direction and handle the movement/deceleration.
	# As good practice, you should replace UI actions with custom gameplay actions.
	var input_dir = Input.get_vector("strafe_left", "strafe_right", "forward", "backward")
	var speed_modifier: float = 1.0 if not is_crouching else crouch_scale
	if Input.is_action_pressed("mode_shift"):
		speed_modifier *= run_scale
		noko_anim.speed_scale = run_scale
	else:
		noko_anim.speed_scale = 1
	var direction = (transform.basis * Vector3(input_dir.x, 0, input_dir.y)).normalized()
	if direction:
		velocity.x = direction.x * SPEED * speed_modifier
		velocity.z = direction.z * SPEED * speed_modifier
	else:
		velocity.x = move_toward(velocity.x, 0, move_slide_scale) #SPEED * speed_modifier)
		velocity.z = move_toward(velocity.z, 0, move_slide_scale) #SPEED * speed_modifier)
	
	move_and_slide()
	if not self.velocity.is_zero_approx() and not is_jumping and not is_kicking:
		anim_play(anim_set["run"])
	elif self.velocity.is_zero_approx() and not is_jumping and not is_kicking:
		anim_play(anim_set["idle"])
	
	if Input.is_action_just_pressed("view"):
		var cam_clipper = cam_base.get_node("SpringArm3D")
		cam_clipper.position.x = -cam_clipper.position.x
	if Input.is_action_just_pressed("crouch"):
		toggle_crouch()
	if Input.is_action_just_pressed("slot_2"):
		activate_slot_2()
	if Input.is_action_just_pressed("special"):
		is_kicking = true
		special_timer.start()
		anim_play("melee_secondary", false, true)
	
	primary_action_control()
	
	return

func primary_action_control():
	if rhand_anchor.get_child_count() < 1:
		return
	if Input.is_action_just_pressed("primary_action") and primary_action_type == U.primary_action_types.TAP:
		rhand_anchor.get_child(0).primary_action(self, primary_action_type)
	if Input.is_action_pressed("primary_action") and primary_action_type == U.primary_action_types.HOLD:
		rhand_anchor.get_child(0).primary_action(self, primary_action_type)
	if Input.is_action_just_released("primary_action") and primary_action_type == U.primary_action_types.HOLD:
		rhand_anchor.get_child(0).stop_action(self, primary_action_type)
	return

func toggle_crouch():
	is_crouching = !is_crouching
	if is_crouching:
		self.global_position.y += 2.5
		cam_base.get_node("SpringArm3D").position.y -= crouch_cam_offset_y
		hitbox_lower_anim.play("half")
		#anim_play(anim_set["crouch"])
		anim_set["idle"] = "crouch"
		anim_set["run"] = "crouch_run"
	if not is_crouching:
		set_armed_type(armed_type)
		self.global_position.y += 1
		cam_base.get_node("SpringArm3D").position.y += crouch_cam_offset_y
		hitbox_lower_anim.play("full")
		anim_play(queued_anims.pop_front())
	return

func anim_play(anim_name: String, is_queued: bool = true, force_anim: bool = false):
	if anim_name == last_anim and not force_anim:
		return
	
	noko_anim.play(anim_name)
	if is_queued:
		queued_anims.push_back(anim_name)
	last_anim = anim_name
	return

func set_armed_type(new_armed_type):
	match new_armed_type:
		armed_types.UNARMED:
			anim_set["idle"] = "unarmed_idle"
			anim_set["run"] = "unarmed_run"
			anim_set["jump"] = "unarmed_jump"
			anim_set["crouch"] = "crouch"
		armed_types.ARMED:
			anim_set["idle"] = "armed_idle"
			anim_set["run"] = "armed_run"
			anim_set["jump"] = "armed_jump"
			anim_set["crouch"] = "crouch"
		armed_types.AKIMBO:
			anim_set["idle"] = "akimbo_idle"
			anim_set["run"] = "akimbo_run"
			anim_set["jump"] = "akimbo_jump"
			anim_set["crouch"] = "crouch"
	armed_type = new_armed_type
	return

func activate_slot_2():
	for rhand_held in rhand_anchor.get_children():
		rhand_held.queue_free()
	var debug_gun_scene := preload("res://Scenes/debug_gun.tscn")
	var debug_gun: Node3D = debug_gun_scene.instantiate()
	rhand_anchor.add_child(debug_gun)
	set_armed_type(armed_types.ARMED)
	return

func _on_debug_ticker_timeout():
	return


func _on_delayed_ticker_timeout():
	# Check for things here if not needed every frame
	if is_jumping and self.is_on_floor():
		is_jumping = false
		anim_play(queued_anims.pop_front())
	return


func _on_special_timer_timeout():
	is_kicking = false
	return
