extends Node

signal net_message_received(message: Dictionary)

# [U]tility
@onready var green_marker_scene: Resource = preload("res://Scenes/green_marker.tscn")
@onready var blue_marker_scene: Resource = preload("res://Scenes/blue_marker.tscn")
@onready var dumb_dropper_scene: Resource = preload("res://Scenes/dumb_dropper.tscn")
@onready var comms_marker_scene: Resource = preload("res://Scenes/comms_marker.tscn")
@onready var sound_ball_scene: Resource = preload("res://Scenes/sound_ball.tscn")
@onready var hurt_ball_scene: Resource = preload("res://Scenes/proximity_hurt_ball.tscn")
@onready var relpos_rig_scene: Resource = preload("res://Scenes/relpos_rig.tscn")
@onready var fade_in_scene: Resource = preload("res://Scenes/fade_in.tscn")
@onready var fade_out_scene: Resource = preload("res://Scenes/fade_out.tscn")

var silly_vector: Vector3 = Vector3(999, 999, 999)

enum primary_action_types {
	TAP,
	HOLD
}

enum hit_types {
	UNDEFINED,
	RAYCAST,
	AREA,
	PROXIMITY,
	BULLET,
	PUNCH,
	KICK,
	KNOCKABLE
}

enum action_events {
	IGNORED,
	MELEE_PUNCH,
	MELEE_HELD,
	SHOOT # ?
}


enum armed_types {
	UNARMED,
	ARMED,
	ARMEDLEFT,
	AKIMBO
}

# FIXME: Adding "ARMEDLEFT" to the above may make the below redundant.
enum hand_states {
	EMPTY,
	SINGLE_LEFT,
	SINGLE_RIGHT,
	AKIMBO
}

enum hands {
	NONE = 0,
	LEFT = 1,
	RIGHT = 2
}

# Need to implement fire selector with mod+reload
enum akimbo_types {
	LINKED, # Primary action button shoots both
	UNLINKED, # Primary action shoots right hand, focus button shoots left hand
	SWITCH # Primary action button shoots right, then left, then right, ...
} # Note for SWITCH: if auto/HOLD, maybe make it so on each release it switches (so it's like left RATATATATATA, right, RATAT, left, RATATATATATATATATA

enum ammo_types {
	NINE_MM,
	FORTY_FIVE,
	FIFTY,
	NATO,
	SEVEN_SIXTY_TWO,
	THREE_FIFTY_SEVEN,
	SHELLS
}

enum weapon_sizes {
	PISTOL,
	MICRO,
	SHORT,
	LONG
}

enum fire_modes {
	NONE,
	SEMI_AUTO,
	FULL_AUTO,
	SINGLE_ACTION,
	OTHER
}

enum weapon_themes {
	LIGHT,
	DARK
}

enum marker_path_types {
	START_POINT,
	END_POINT
}

enum net_message_types {
	VAR # {variable_name: variable_value, ...}
}

func is_true(array_item):
	return array_item == true

#func fs_write_text(fpath: String, new_text: String):
#	var file: FileAccess = FileAccess.open(fpath, FileAccess.WRITE)
#	file.store_string(new_text)
#	return
#
#func fs_read_text(fpath: String):
#	var file = FileAccess.open(fpath, FileAccess.READ)
#	var content = file.get_as_text()
#	return content

#func save(content):
#    var file = FileAccess.open("user://save_game.dat", FileAccess.WRITE)
#    file.store_string(content)
#
#func load():
#    var file = FileAccess.open("user://save_game.dat", FileAccess.READ)
#    var content = file.get_as_text()
#    return content

func world_raycast(caller: Node3D, from: Vector3, to: Vector3, exclusions: Array[RID]) -> Dictionary:
	var space = caller.get_world_3d().direct_space_state
	var ray_query = PhysicsRayQueryParameters3D.new()
	ray_query.from = from
	ray_query.to = to
	ray_query.collide_with_areas = false
	ray_query.exclude = exclusions
	#ray_query.hit_from_inside = true
	return space.intersect_ray(ray_query)

func spawn_comms_marker(attach_to: Node3D, spawn_position: Vector3, duration: float = 1.0):
	var comms_marker: Node3D = comms_marker_scene.instantiate()
	comms_marker.duration = duration
	attach_to.add_child(comms_marker)
	comms_marker.global_position = spawn_position
	return comms_marker

func spawn_green_marker(attach_to: Node3D, spawn_position: Vector3, duration: float = 1.0):
	var green_marker: Node3D = green_marker_scene.instantiate()
	green_marker.duration = duration
	attach_to.add_child(green_marker)
	green_marker.global_position = spawn_position
	return green_marker

func spawn_blue_marker(attach_to: Node3D, spawn_position: Vector3, duration: float = 1.0):
	var blue_marker: Node3D = blue_marker_scene.instantiate()
	blue_marker.duration = duration
	attach_to.add_child(blue_marker)
	blue_marker.global_position = spawn_position
	return blue_marker


func spawn_dumb_dropper(attach_to: Node3D, spawn_position: Vector3, speed: float = 1.0):
	var dumb_dropper: Node3D = dumb_dropper_scene.instantiate()
	attach_to.add_child(dumb_dropper)
	dumb_dropper.dropping_speed = speed
	dumb_dropper.global_position = spawn_position
	return dumb_dropper

func spawn_sound_ball(attach_to: Node3D, spawn_position: Vector3, sound_distance: float = 0.0, sound_tags: String = ""):
	var sound_ball: Node3D = sound_ball_scene.instantiate()
	attach_to.add_child(sound_ball)
	sound_ball.global_position = spawn_position
	if sound_distance:
		sound_ball.sound_distance = sound_distance
	if sound_tags:
		sound_ball.tags = sound_tags
	return sound_ball

func spawn_sound_player(attach_to: Node3D, spawn_position: Vector3, stream_path: String, lifetime: float = 5.0):
	var sound_player: AudioStreamPlayer3D = AudioStreamPlayer3D.new()
	sound_player.stream = load(stream_path)
	attach_to.add_child(sound_player)
	sound_player.global_position = spawn_position
	get_tree().create_tween().tween_callback(
		func ():
			if not sound_player:
				return
			sound_player.queue_free()
	).set_delay(lifetime)
	return sound_player

func spawn_hurt_ball(attach_to: Node3D, spawn_position: Vector3):
	var hurt_ball: Node3D = hurt_ball_scene.instantiate()
	attach_to.add_child(hurt_ball)
	hurt_ball.global_position = spawn_position
	return hurt_ball

func spawn_relpos_rig(attach_to: Node3D, duration: float = 0.0):
	var relpos_rig: Node3D = relpos_rig_scene.instantiate()
	relpos_rig.die_time = duration
	attach_to.add_child(relpos_rig)
	return relpos_rig

func get_closest_position(target_position: Vector3, source_positions: Array):
	var closest_position: Vector3
	var lowest_distance: float = -1.0
	for source_position in source_positions:
		var distance: float = target_position.distance_to(source_position)
		if lowest_distance == -1:
			lowest_distance = distance
			closest_position = source_position
			continue
		if distance < lowest_distance:
			lowest_distance = distance
			closest_position = source_position
		continue
	return closest_position

func set_mesh_materials_unshaded(mesh: MeshInstance3D):
	for i in range(mesh.mesh.get_surface_count()):
		var current_material: Material = mesh.get_active_material(i)
		if current_material == null:
			return
		current_material.shading_mode = 0
	return

func random_choice(choice_array: Array):
	var array_length: int = len(choice_array)
	var lower_bound: int = 0
	var upper_bound: int = array_length - 1 if array_length > 0 else 0
	return choice_array[randi_range(lower_bound, upper_bound)]

func random_choice_or_nothing(choice_array: Array, chance: float = 0.5):
	if randf_range(0.0, 0.99) < chance:
		return null
	var array_length: int = len(choice_array)
	var lower_bound: int = 0
	var upper_bound: int = array_length - 1 if array_length > 0 else 0
	return choice_array[randi_range(lower_bound, upper_bound)]

func fade_in_screen(attach_to: Node):
	var fade_in: Control = fade_in_scene.instantiate()
	attach_to.add_child(fade_in)
	return fade_in

func fade_out_screen(attach_to: Node, speed: float = 0.0):
	var fade_out: Control = fade_out_scene.instantiate()
	if speed > 0.0:
		fade_out.fade_speed = speed
	attach_to.add_child(fade_out)
	return fade_out


func send_net_message(recipient: int, message: Dictionary):
	rpc_id(recipient, "receive_net_message", message)
	return

func broadcast_net_message(message: Dictionary):
	rpc("receive_net_message", message)
	return

@rpc("any_peer", "call_local")
func receive_net_message(message: Dictionary):
	net_message_received.emit(message)
	return

func is_networked():
	return is_instance_of(multiplayer.multiplayer_peer, ENetMultiplayerPeer)

func mouse_motion_to_dict(raw_event: InputEventMouseMotion):
	return {
		"button_mask": raw_event.button_mask,
		"position": raw_event.position,
		"relative": raw_event.relative,
		"velocity": raw_event.velocity,
		"pressure": raw_event.pressure,
		"tilt": raw_event.tilt,
		"pen_inverted": raw_event.pen_inverted
	}

func dict_to_mouse_motion(event_dict: Dictionary):
	var mouse_motion_event: InputEventMouseMotion = InputEventMouseMotion.new()
	for prop in event_dict.keys():
		mouse_motion_event.set(prop, event_dict[prop])
	return mouse_motion_event

func net_randf_range(from: float, to: float):
	return

func net_randi_range(from: int, to: int):
	return

func cycled_array_index(arr: Array, idx: int):
	var next_index: int
	var current_index: int = idx
	var max_index: int = len(arr) - 1
	if (current_index + 1) > max_index:
		next_index = 0
	else:
		next_index = current_index + 1
	return next_index

func is_node_root(node: Node):
	return str(node.get_path()) == "/root"

func coin_flip():
	var heads: bool = randi_range(0, 1)
	return 1 if heads else 0

func read_propstring(datastring: String):
	var propsign: String = "#"
	var flat_datastring: String = datastring.to_lower()
	var props: Dictionary = {}
	var propchunks: Array = flat_datastring.split("#")
	#for propchunk in flat_datastring.split("#"):
	for propchunk in propchunks:
		if not propchunk.contains("="):
			# Skip node name, lazily
			continue
		var prop_tokens: Array = propchunk.split("-")[0].split("=")
		var prop_key: String = prop_tokens[0]
		if not propchunk.contains("="):
			props[propchunk] = ""
			continue
		var prop_value: String = prop_tokens[1].replace("minus", "-")
		props[prop_key] = prop_value
	
	return props

func spawn_bottle_break(effect_position: Vector3, effect_props: Dictionary):
	var world_effect: Node3D = load("res://Scenes/bottle_break_effect.tscn").instantiate()
	Blackboard.current_world.add_child(world_effect)
	world_effect.global_position = effect_position
	if effect_props.has("bottletype") and effect_props["bottletype"] == "ketchup":
		world_effect.is_ketchup = true
	world_effect.activate()
	return

func distance(from_node: Node3D, to_node: Node3D):
	return from_node.global_position.distance_to(to_node.global_position)

func grptop(group_name: String):
	return get_tree().get_first_node_in_group(group_name)

func drop_mag(mag_mesh: MeshInstance3D):
	var world_mag: MeshInstance3D = mag_mesh.duplicate()
	Blackboard.current_world.add_child(world_mag)
	world_mag.global_position = mag_mesh.global_position
	world_mag.visible = true
	var tween_drop_y: Tween = get_tree().create_tween()
	var tween_rotate: Tween = get_tree().create_tween()
	var drop_destination: Vector3 = world_mag.global_position + Vector3(0, -5, 0)
	var rotate_destination: Vector3 = world_mag.global_rotation_degrees + Vector3(randf_range(-2, 2), 0, randf_range(-2, 2))
	tween_drop_y.tween_property(world_mag, "global_position", drop_destination, .8).set_trans(Tween.TRANS_SINE)
	tween_rotate.tween_property(world_mag, "global_rotation", rotate_destination, .8).set_trans(Tween.TRANS_CUBIC)
	
	await tween_drop_y.finished
	await tween_rotate.finished
	world_mag.queue_free()
	return

func spawn_world_weapon(weapon_name: String, spawn_position: Vector3):
	#var spawn_weapon: String = "beretta_nine" #Temp?
	var world_item: RigidBody3D = Weapons.by_name(weapon_name)["world_item"].instantiate()
	Blackboard.current_world.add_child(world_item)
	world_item.global_position = spawn_position
	world_item.rotation_degrees.y = 180
	
	#for p in get_tree().get_nodes_in_group("players"):
	for a in get_tree().get_nodes_in_group("actors"):
		world_item.add_collision_exception_with(a) # This shouldn't go here!
	for d in get_tree().get_nodes_in_group("droppables"):
		world_item.add_collision_exception_with(d)
	
	world_item.linear_velocity = Vector3(0, 10, 0)
	world_item.angular_velocity = Vector3(0, 0, 5)
	return world_item

func is_same_signf(num_a: float, num_b: float):
	return (num_a * num_b) >= 0

func get_aabb_global_endpoints(mesh_instance: MeshInstance3D) -> Array:
	# https://www.reddit.com/r/godot/comments/ooln68/i_need_to_get_the_global_position_of_edges_of_3d/?rdt=55047
	if not is_instance_valid(mesh_instance):
		return []

	var mesh: Mesh = mesh_instance.mesh
	if not mesh:
		return []

	var aabb: AABB = mesh.get_aabb()
	var global_endpoints := []
	for i in range(8):
		var local_endpoint: Vector3 = aabb.get_endpoint(i)
		var global_endpoint: Vector3 = mesh_instance.to_global(local_endpoint)
		global_endpoints.push_back(global_endpoint)
	return global_endpoints


func get_aabb_global_endpoints_rich(mesh_instance: MeshInstance3D) -> Dictionary:
	# https://www.reddit.com/r/godot/comments/ooln68/i_need_to_get_the_global_position_of_edges_of_3d/?rdt=55047
	if not is_instance_valid(mesh_instance):
		return {}

	var mesh: Mesh = mesh_instance.mesh
	if not mesh:
		return {}

	var aabb: AABB = mesh.get_aabb()
	var global_endpoints: Dictionary = {}
	for i in range(8):
		var local_endpoint: Vector3 = aabb.get_endpoint(i)
		var global_endpoint: Vector3 = mesh_instance.to_global(local_endpoint)
		#global_endpoints.push_back(global_endpoint)
		global_endpoints[i] = global_endpoint
	return global_endpoints

func set_model_material(mesh_host: Node3D, new_material: Material, uv_scale: float = 1.0) -> void:
	for mesh in mesh_host.find_children("*", "MeshInstance3D"):
		mesh.set_surface_override_material(0, new_material)
		mesh.get_surface_override_material(0).uv1_scale *= uv_scale
	return

func set_mesh_materials_alpha(mesh: MeshInstance3D):
	for i in range(mesh.mesh.get_surface_count()):
		var current_material: Material = mesh.get_active_material(i)
		if current_material == null:
			return
		current_material.transparency = 1
		#alpha_mesh.get_active_material(0).transparency = 1
	return

func get_weapon_raycast(weapon_node: Node3D) -> RayCast3D:
	var guessed_raycast: RayCast3D
	if weapon_node.get("raycast"):
		guessed_raycast = weapon_node.raycast
	elif weapon_node.get("main_raycast"):
		guessed_raycast = weapon_node.main_raycast
	return guessed_raycast

func look_at2() -> void:
	# Do a 3D look_at but use 2D funcs/coords for a possibly simpler, more reliable, and faster(?) look_at
	# (without having to use eyes and stuff, even?)
	# (without the C++ error origin/target alignment error, even?)
	return
