extends Control

# TODO
# - Animation preview
# - Option to set as default animation
# - Somehow make an actor out of it (wiseguy, merc_sara, shoot, melee)
# - Player compatible (will have to code on player side too)

@onready var spatials: Node3D = $Spatials
@onready var choose_model_filedialog: FileDialog = $choose_model_filedialog
var chosen_model_path: String = ""
var viewport_model: Node3D
var viewport_model_skeleton: Skeleton3D

@onready var actor_name_input: TextEdit = $actor_name
@onready var select_meshes: Button = $select_meshes
@onready var mesh_list: VBoxContainer = $mesh_list
@onready var mesh_list_checkbox_template: CheckBox = $mesh_list/checkbox_template
@onready var mesh_list_scrollbar: VScrollBar = $mesh_list_scroll
var mesh_node_list: Dictionary
var mesh_list_base_y_position: float = 26.0

var mesh_updates_blocked: bool = false

func _ready():
	return


# Called every frame. 'delta' is the elapsed time since the previous frame.
func _process(delta):
	lazy_mesh_visibility_update()
	return

func add_mesh_list_item(new_item: MeshInstance3D):
	var new_mesh_list_item: CheckBox = mesh_list_checkbox_template.duplicate()
	new_mesh_list_item.text = new_item.name
	mesh_list.add_child(new_mesh_list_item)
	new_mesh_list_item.button_pressed = false
	new_mesh_list_item.visible = true
	return

func populate_mesh_selection():
	if not viewport_model_skeleton:
		return
	
	mesh_list.visible = true
	mesh_list_scrollbar.visible = true
	for mesh in viewport_model_skeleton.get_children():
		mesh_node_list[mesh.name] = mesh
		add_mesh_list_item(mesh)
	for checkbox in mesh_list.get_children():
		if checkbox.name.contains("template"):
			continue
		checkbox.button_pressed = mesh_node_list[checkbox.text].visible
	return

func depopulate_mesh_selection():
	mesh_list.visible = false
	mesh_list_scrollbar.visible = false
	mesh_node_list.clear()
	for c in mesh_list.get_children():
		if c.name.contains("template"):
			continue
		c.queue_free()
	return

func lazy_mesh_visibility_update():
	if mesh_updates_blocked:
		return
	for c in mesh_list.get_children():
		if c.name.contains("template"):
			continue
		
		if c.text not in mesh_node_list:
			return
		var mesh: MeshInstance3D = mesh_node_list[c.text]
		mesh.visible = c.button_pressed
	return

func _on_load_model_pressed():
	#var model_file: Resource = preload("res://Staging/Models/wiseguy_v2.glb")
	depopulate_mesh_selection()
	select_meshes.button_pressed = false
	
	mesh_updates_blocked = true
	if viewport_model:
		viewport_model.free()
	if not chosen_model_path:
		prints(self.name, "no model path:", chosen_model_path)
		return
	var model_file: Resource = load(chosen_model_path)
	viewport_model = model_file.instantiate()
	viewport_model_skeleton = viewport_model.find_child("Skeleton3D", true)
	spatials.add_child(viewport_model)
	if $flip.button_pressed:
		viewport_model.rotation_degrees.y += 180
	mesh_updates_blocked = false
	return


func _on_choose_model_pressed():
	choose_model_filedialog.show()
	return


func _on_choose_model_filedialog_file_selected(path):
	chosen_model_path = path
	return


func _on_mesh_list_child_exiting_tree(node):
	#debug
	prints(self.name, "child exited", node)
	return


func _on_select_meshes_toggled(toggled_on):
	if toggled_on:
		populate_mesh_selection()
		return
	if not toggled_on:
		depopulate_mesh_selection()
		return
	return


func _on_mesh_list_scroll_value_changed(value):
	mesh_list.position.y = lerp(mesh_list_base_y_position, -(mesh_list_base_y_position * 2), value)
	return


func _on_mesh_list_gui_input(event):
	# Might be able to use the scroll wheel
	return


func _on_export_actor_pressed():
	if not actor_name_input.text:
		$messages.text = "No name given."
		return
	
	if not chosen_model_path:
		$messages.text = "No model chosen."
		return
	
	var new_actor_respath: String = "res://Scenes/" + actor_name_input.text + ".tscn"
	
	#var new_actor: CharacterBody3D = load(actor_master_respath).instantiate()
	var new_actor: CharacterBody3D = CharacterBody3D.new()
	self.add_child(new_actor)
	new_actor.owner = self
	new_actor.name = actor_name_input.text
	
	configure_actor(new_actor, chosen_model_path)
	
	var new_actor_scene: PackedScene = PackedScene.new()
	new_actor_scene.pack(new_actor)
	ResourceSaver.save(new_actor_scene, new_actor_respath)
	$messages.text = "Saved " + new_actor_respath
	#ResourceSaver.save(new_actor_scene, new_actor_respath)
	return




# ------------------------------------------------------------
var bone_attachment_mappings: Dictionary = {
	"attachment_head": "head",
	"attachment_chest": "chest",
	"attachment_hand_L": "hand.L",
	"attachment_hand_R": "hand.R",
	"attachment_pelvis": "pelvis",
	"attachment_lower_leg_L": "lower_leg.L",
	"attachment_lower_leg_R": "lower_leg.R",
	"attachment_upper_leg_L": "upper_leg.L",
	"attachment_upper_leg_R": "upper_leg.R",
	"attachment_foot_L": "foot.L",
	"attachment_foot_R": "foot.R"
}

var hitbox_names: Array[String] = [
	"hitbox_head",
	"hitbox_chest",
	"hitbox_pelvis",
	"hitbox_lower_leg_L",
	"hitbox_lower_leg_R",
	"hitbox_upper_leg_L",
	"hitbox_upper_leg_R",
	"hitbox_foot_L",
	"hitbox_foot_R"
]

func create_hitbox_colliders(actor_root: CharacterBody3D, hitbox_names: Array):
	for hitbox_name in hitbox_names:
		var base_name: String = hitbox_name.replace("hitbox_", "")
		var new_hitbox: CollisionShape3D = CollisionShape3D.new()
		new_hitbox.name = hitbox_name
		#new_hitbox.shape = BoxShape3D.new()
		var part_name: String = base_name.replace("_L", "").replace("_R", "")
		var colshape_respath: String = "res://Resources/hitbox_default_" + part_name + ".shape"
		new_hitbox.shape = load(colshape_respath)
		actor_root.add_child(new_hitbox)
		new_hitbox.owner = actor_root
	return

func create_bone_attachments(actor_root: CharacterBody3D, skeleton: Skeleton3D, names_and_bones: Dictionary):
	for attachment_name in names_and_bones.keys():
		var new_bone_attachment: BoneAttachment3D = BoneAttachment3D.new()
		new_bone_attachment.name = attachment_name
		new_bone_attachment.bone_name = names_and_bones[attachment_name]
		skeleton.add_child(new_bone_attachment)
		new_bone_attachment.owner = actor_root
	return

func create_remote_transforms(actor_root: CharacterBody3D, skeleton: Skeleton3D, hitbox_names: Array):
	for hitbox_name in hitbox_names:
		var hitbox_node: CollisionShape3D = actor_root.find_child(hitbox_name)
		var base_name: String = hitbox_name.replace("hitbox_", "")
		var attachment_name: String = "attachment_" + base_name
		var attachment_node: BoneAttachment3D = skeleton.find_child(attachment_name)
		var new_remote_transform: RemoteTransform3D = RemoteTransform3D.new()
		attachment_node.add_child(new_remote_transform)
		new_remote_transform.name = "remote_" + base_name
		if base_name.contains("lower_leg"):
			new_remote_transform.position.y = 0.788
		elif base_name.contains("upper_leg"):
			new_remote_transform.position.y = 0.987
		elif base_name.contains("foot"):
			new_remote_transform.position.y = 0.462
		new_remote_transform.owner = actor_root
		new_remote_transform.remote_path = new_remote_transform.get_path_to(hitbox_node)
	return

func create_model(actor_root: CharacterBody3D, respath: String, new_name: String = "", is_flipped: bool = false):
	var new_model: Node3D = load(respath).instantiate()
	actor_root.add_child(new_model)
	new_model.owner = actor_root
	if new_name:
		new_model.name = new_name
	if is_flipped:
		new_model.rotation_degrees.y = -180
	return new_model

func create_actor_script(actor_root: CharacterBody3D, template_script_respath: String):
	var dir: DirAccess = DirAccess.open("res://Scripts/")
	var new_script_path: String = "res://Scripts/" + actor_root.name + ".gd"
	dir.copy("res://Scripts/actor_characterbody_master.gd", new_script_path)
	actor_root.set_script(load(new_script_path))
	return

func create_spatial_eyes(actor_root: CharacterBody3D):
	var new_mid_anchor: Node3D = Node3D.new()
	new_mid_anchor.name = "mid_anchor"
	new_mid_anchor.position.y = 3.966
	actor_root.add_child(new_mid_anchor)
	new_mid_anchor.owner = actor_root
	
	var new_mid_eyes: Node3D = Node3D.new()
	new_mid_eyes.name = "eyes"
	new_mid_anchor.add_child(new_mid_eyes)
	new_mid_eyes.owner = actor_root
	
	
	var new_base_anchor: Node3D = Node3D.new()
	new_base_anchor.name = "base_anchor"
	actor_root.add_child(new_base_anchor)
	new_base_anchor.owner = actor_root
	
	var new_base_eyes: Node3D = Node3D.new()
	new_base_eyes.name = "eyes"
	new_base_anchor.add_child(new_base_eyes)
	new_base_eyes.owner = actor_root
	return

func create_raycasts(actor_root: CharacterBody3D):
	var sight_raycast: RayCast3D = RayCast3D.new()
	sight_raycast.name = "sight_raycast"
	var strike_raycast: RayCast3D = RayCast3D.new()
	strike_raycast.name = "strike_raycast"
	
	for raycast in [sight_raycast, strike_raycast]:
		actor_root.add_child(raycast)
		var approx_chest_height: float = 5.325
		raycast.position.y = approx_chest_height
		raycast.target_position.z = -200
		raycast.owner = actor_root
	return

func create_nav_agent(actor_root: CharacterBody3D):
	var new_nav_agent: NavigationAgent3D = NavigationAgent3D.new()
	actor_root.add_child(new_nav_agent)
	new_nav_agent.name = "NavigationAgent3D"
	new_nav_agent.owner = actor_root
	return

func create_debug_label(actor_root: CharacterBody3D):
	var debug_label: Label3D = Label3D.new()
	debug_label.position.y = 7.5
	debug_label.name = "debug_label"
	debug_label.text = "..."
	debug_label.billboard = 1
	debug_label.pixel_size = 0.01
	actor_root.add_child(debug_label)
	debug_label.owner = actor_root
	return

func assign_groups(actor_root: CharacterBody3D):
	actor_root.add_to_group("hit_takers", true)
	actor_root.add_to_group("actors", true)
	return

func sync_mesh_visibility(target_model: Node3D, source_model: Node3D):
	var target_skel: Skeleton3D = target_model.get_node("Armature/Skeleton3D")
	var source_skel: Skeleton3D = source_model.get_node("Armature/Skeleton3D")
	
	for source_mesh in source_skel.get_children():
		var target_mesh: MeshInstance3D = target_skel.get_node(NodePath(source_mesh.name))
		target_mesh.visible = source_mesh.visible
	return

func configure_actor(actor_root: CharacterBody3D, model_respath: String):
	var template_script_respath: String = "res://Scripts/actor_characterbody_master.gd"
	var model: Node3D = create_model(actor_root, model_respath, "model", true)
	var skel: Skeleton3D = model.find_child("Skeleton3D")
	
	actor_root.set_editable_instance(model, true)
	
	create_hitbox_colliders(actor_root, hitbox_names)
	create_bone_attachments(actor_root, skel, bone_attachment_mappings)
	create_remote_transforms(actor_root, skel, hitbox_names)
	create_actor_script(actor_root, template_script_respath)
	create_spatial_eyes(actor_root)
	create_nav_agent(actor_root)
	create_raycasts(actor_root)
	create_debug_label(actor_root)
	assign_groups(actor_root)
	sync_mesh_visibility(model, viewport_model)
	return


func _on_close_pressed():
	self.queue_free()
	return
