extends CharacterBody3D

@onready var model: Node3D = $ggbot_crown_edit

@export var is_headlights_allowed: bool = true
@export var spawn_with_headlights: bool = false

@onready var wheel_front_left: MeshInstance3D = $ggbot_crown_edit/wheel_front_left
@onready var wheel_back_left: MeshInstance3D = $ggbot_crown_edit/wheel_back_left
@onready var wheel_front_right: MeshInstance3D = $ggbot_crown_edit/wheel_front_right
@onready var wheel_back_right: MeshInstance3D = $ggbot_crown_edit/wheel_back_right
@onready var floor_sensor: RayCast3D = $floor_sensor
@onready var floor_sensor_front: RayCast3D = $floor_sensor_front
@onready var floor_sensor_back: RayCast3D = $floor_sensor_back
@onready var chasecam_base: Node3D = $chasecam_base
@onready var chasecam: Camera3D = $chasecam_base/chasecam
@onready var chasecam_reset_marker: Marker3D = $chasecam_base/chasecam_reset_marker
@onready var carrot: Marker3D = $carrot_base/carrot
@onready var rot_dummy: RigidBody3D = $rot_dummy
#@onready var remote_xform_rot_dummy: RemoteTransform3D = $remote_xform_rot_dummy
var original_wheel_rotations: Dictionary = {}

@onready var active_cameras: Array = [
	$chasecam_base/chasecam,
	$mounted_cameras/mounted_camera_2,
	$mounted_cameras/mounted_camera_1
]
var active_camera_index: int = 0

@onready var sound_container: Node3D = $Sounds
var is_input_allowed: bool = false
var driver_parent: Node3D
var driver: Node3D
var driver_rotation: Vector3
var is_claim_allowed: bool = true
var is_eject_allowed: bool = false
var is_recently_on_floor: bool = false
var is_reversing: bool = false
var is_accelerating: bool = false
var is_steering: bool = false

var max_accel: float = 100.0
var accel_input: Vector2 = Vector2.ZERO
var steer_input: Vector2 = Vector2.ZERO
var steer_speed: float = 2.0
var steer_downscale: float = 0.05
var steer_amount: float = 0.0
var max_steer_amount: float = 3.0
var accel_force: float = 0.15
var active_accel: float = 0.0
var mouse_sensitivity: float = 0.0

var default_slope_interp_speed: float = 2.0
var max_slope_interp_speed: float = 7.0
var slope_interp_speed: float = 2.0

@onready var passenger_seat_marker: Marker3D = $seat_positions/passenger_seat
var initial_xform: Transform3D
var shared_delta: float = 0.0
var last_force_vec: Vector3 = Vector3.ZERO
var last_steer_amount: float = 0.0
var last_accel_dir: float = 0

func _ready() -> void:
	mouse_sensitivity = Global.parse_mouse_sensitivity()
	initial_xform = global_transform
	init_wheels()
	await get_tree().process_frame
	init_rot_dummy()
	return

func _unhandled_input(event: InputEvent) -> void:
	if not is_input_allowed:
		return
	if event is InputEventMouseMotion:
		chasecam_base.rotation.x -= (event.relative.y * mouse_sensitivity)
		chasecam_base.rotation_degrees.x = clamp(chasecam_base.rotation_degrees.x, -180, 180)
		chasecam_base.rotation.y -= (event.relative.x * mouse_sensitivity)
	return

func _physics_process(delta: float) -> void:
	$debug_control/debug_label.text = \
		str(get_vlen()) + "\n" + \
		"floor: " + str(is_on_floor())# + "\n" + \
	shared_delta = delta
	
	rot_dummy.global_position = self.global_position
	self.global_rotation = rot_dummy.global_rotation
	
	if is_on_floor():
		is_recently_on_floor = true
		get_tree().create_tween().tween_callback(func (): if is_instance_valid(self) and not is_on_floor(): is_recently_on_floor = false).set_delay(.7)
	else:
		A.apply_gravity(self, delta * 3.0)
		
		
		#global_rotation.x += global_rotation.x * .02
		#global_rotation.z += global_rotation.x * .02
	
	if is_input_allowed:
		input_control()
	
	accel_control()
	steer_control()
	#floor_align_control()
	if not velocity.is_zero_approx():
		move_and_slide()
		drag_velo()
	
	return

########

func input_control() -> void:
	if Input.is_action_just_pressed("claim") and is_eject_allowed:
		eject_driver()
		return
	
	accel_input = Vector2.ZERO
	
	is_accelerating = Input.is_action_pressed("forward")
	is_reversing = Input.is_action_pressed("backward")
	is_steering = Input.is_action_pressed("strafe_left") or Input.is_action_pressed("strafe_right")
	
	#if Input.is_action_pressed("forward"):
	if is_accelerating:
		accel_input += Vector2.UP
	#if Input.is_action_pressed("backward"):
	if is_reversing:
		accel_input += Vector2.DOWN
	
	if accel_input.y != 0.0 and last_accel_dir != accel_input.y:
		active_accel = 0.0
	
	last_accel_dir = accel_input.y
	
	steer_input = Vector2.ZERO
	if Input.is_action_pressed("strafe_left"):
		steer_input += Vector2.LEFT
	if Input.is_action_pressed("strafe_right"):
		steer_input += Vector2.RIGHT
	
	return

func accel_control() -> void:
	if not accel_input or get_vlen() > max_accel:
		if absf(active_accel) >= accel_force:
			active_accel -= accel_force
		else:
			active_accel = 0.0
		
		if absf(last_force_vec.length()) > 1.0 and absf(steer_amount) > 1.0:
			velocity += (last_force_vec * .5)
			last_force_vec = (last_force_vec * .5)
			if absf(last_force_vec.length()) < 1.0:
				last_force_vec = Vector3.ZERO
		return
	
	
	active_accel += accel_force
	var straight_force_vec: Vector3 = Vector3(
		accel_input.x,
		0,
		accel_input.y
	) * active_accel
	#var force_vec: Vector3 = straight_force_vec.rotated(Vector3.UP, self.global_rotation.y)
	var force_vec: Vector3 = straight_force_vec.rotated(Vector3.UP, self.global_rotation.y + deg_to_rad(steer_amount * 10.0))
	velocity += force_vec
	last_force_vec = force_vec
	return

func steer_control() -> void:
	#self.global_rotation_degrees.y -= steer_input.x * steer_speed
	if steer_input:
		steer_amount += (steer_input.x * steer_speed) * steer_downscale
		steer_amount = clampf(steer_amount, -max_steer_amount, max_steer_amount)
	elif absf(steer_amount) > 0.1:
		steer_amount = lerpf(steer_amount, 0.0, .08)
	else:
		steer_amount = 0.0
	
	if steer_amount:
		var applied_steer: float = steer_amount * last_accel_dir
		if rot_dummy.process_mode == Node.PROCESS_MODE_DISABLED:
			self.global_rotation_degrees.y += applied_steer
		else:
			rot_dummy.global_rotation_degrees.y += applied_steer
	
	$debug_control/debug_label.text += "\n steer: " + str(steer_amount)
	return

func floor_align_control() -> void:
	var accel_ratio: float = get_vlen() / max_accel
	var floor_sensor_collider: Object = floor_sensor.get_collider()
	if floor_sensor_collider:
		var new_xform: Transform3D = align_with_y(global_transform, floor_sensor.get_collision_normal())
		global_transform = global_transform.interpolate_with(new_xform, slope_interp_speed * shared_delta)
	elif not floor_sensor_collider and (is_on_floor() or is_recently_on_floor):
		var new_xform: Transform3D = align_with_y(global_transform, get_floor_normal())
		global_transform = global_transform.interpolate_with(new_xform, slope_interp_speed * shared_delta)
	return

############

func claimed_by(new_driver: Node3D):
	if not is_claim_allowed:
		return
	is_claim_allowed = false
	if driver:
		prints(self.name, "Already a driver, claim aborted.")
		return
	
	get_tree().create_tween().tween_callback(func (): if is_instance_valid(self): is_eject_allowed = true).set_delay(.5)
	
	driver = new_driver
	driver_rotation = driver.global_rotation
	driver_parent = driver.get_parent()
	self.add_collision_exception_with(driver)
	driver.mount_vehicle(self)
	driver.global_position = $seat_positions/driver_seat.global_position
	#driver.global_rotation = $seat_positions/driver_seat.global_rotation * Vector3(1, 0, 1)
	driver.global_rotation = self.global_rotation
	driver.reparent(self)
	is_input_allowed = true
	await get_tree().create_timer(2.0).timeout
	is_claim_allowed = true
	return

func activate_camera():
	active_cameras[active_camera_index].make_current()
	return

func eject_driver():
	driver.dismount_vehicle()
	driver.global_position = $dismount_positions/driver_seat.global_position
	driver.velocity.y += 10.0
	driver.global_rotation = driver_rotation
	is_input_allowed = false
	
	driver.reparent(driver_parent)
	await get_tree().process_frame
	self.remove_collision_exception_with(driver)
	driver = null
	driver_parent = null
	is_eject_allowed = false
	return

func set_passenger(actor: PhysicsBody3D) -> void:
	actor.global_position = passenger_seat_marker.global_position
	actor.global_rotation = passenger_seat_marker.global_rotation
	return

func get_driver_dismount_position() -> Vector3:
	return $dismount_positions/driver_seat.global_position

func get_passenger_dismount_position() -> Vector3:
	return $dismount_positions/passenger_seat.global_position

func init_wheels() -> void:
	for wheel in [wheel_front_left, wheel_back_left, wheel_front_right, wheel_back_right]:
		original_wheel_rotations[wheel] = wheel.rotation_degrees
	return


func _on_ticker_timeout() -> void:
	return

func drag_velo(threshold: float = 0.1) -> void:
	if absf(velocity.length()) < threshold:
		velocity = Vector3.ZERO
		return
	velocity -= velocity * .1
	return

func get_vlen() -> float:
	return velocity.length()

func align_with_y(xform: Transform3D, new_y: Vector3) -> Transform3D:
	# Stolen from https://kidscancode.org/godot_recipes/4.x/3d/3d_align_surface/index.html
	xform.basis.y = new_y
	xform.basis.x = -xform.basis.z.cross(new_y)
	xform.basis = xform.basis.orthonormalized()
	return xform

func init_rot_dummy() -> void:
	rot_dummy.add_collision_exception_with(self)
	for a in get_tree().get_nodes_in_group("actors"):
		if not is_instance_valid(a):
			continue
		if not is_instance_of(a, PhysicsBody3D):
			continue
		if a == rot_dummy:
			continue
		rot_dummy.add_collision_exception_with(a)
	rot_dummy.reparent(Blackboard.current_world, true)
	#remote_xform_rot_dummy.remote_path = remote_xform_rot_dummy.get_path_to(rot_dummy)
	rot_dummy.process_mode = Node.PROCESS_MODE_INHERIT
	return
