extends CharacterBody3D

@export var logic_priority: bool = true

var main_group_name: String = "cultists"
const JUMP_VELOCITY = 4.5
const BASE_MOVE_SPEED = 11.0
const MAX_VELO: float = BASE_MOVE_SPEED * 1.5
var move_speed: float = BASE_MOVE_SPEED
var movement_threshold: float = 1.0
var run_scale = 1.2


@onready var knife_mesh: MeshInstance3D = $model/Armature/Skeleton3D/Cube

@onready var eyes: Node3D = self.get_node("mid_anchor/eyes")
@onready var debug_label: Label3D = self.get_node("debug_label")
@onready var anim: AnimationPlayer = self.get_node("model/AnimationPlayer")
@onready var sight_raycast: RayCast3D =  self.get_node("sight_raycast")
@onready var strike_raycast: RayCast3D = self.get_node("strike_raycast")
@onready var nav_agent: NavigationAgent3D = self.get_node("NavigationAgent3D")
@onready var rhand_anchor: Node3D = $model/Armature/Skeleton3D/attachment_hand_R/anchor
@onready var hitboxes: Array = [
	$hitbox_head,
	$hitbox_chest,
	$hitbox_pelvis,
	$hitbox_lower_leg_L,
	$hitbox_lower_leg_R,
	$hitbox_upper_leg_L,
	$hitbox_upper_leg_R,
	$hitbox_foot_L,
	$hitbox_foot_R
]

@export var performance_mode: bool = false
@export var drop_on_death: PackedScene

@export var health: int = 35
var original_health: int = health
@export var spawn_weapon: String = ""
@export var melee_damage: float = 16.0
@export var melee_recoil: float = -30.0
@export var equip_delay_time: float = 1.0
@export var alert_time: float = 0.25
@export var shoot_intent_time: float = .7
@export var attack_time: float = .80
@export var enable_sound_following: bool = true
var is_awaiting_alert_time: bool = false
var is_dead: bool = false
var is_expired: bool = false

var active_weapon_name: String
var active_weapon_node: Node3D
var active_weapon_ammo_left: int
var active_weapon_mag_size: int
var is_equipping: bool = false
var is_knife_equipped: bool = false
var is_aiming: bool = false
var is_attacking: bool = false
var is_melee_attacking: bool = false
var is_flinching: bool = false
var is_alert_moving: bool = false
var is_strafing_enabled: bool = false
var animation_locked: bool = false

var anim_set: Dictionary = {
	"idle": "unarmed_idle",
	"run": "unarmed_run",
	"equip": "draw_pistol2",
	"die": "cult_die"
}

var ouch_sounds: Array = []
var die_sounds: Array = []
var stab_speak_sounds: Array = []
var welcome_speak_sounds: Array = []
var knife_swipe_sounds: Array = []

var behavior_process_ticker: float = 0.0
const FAST_PROCESS_TICK: float = 0.049
const DEFAULT_PROCESS_TICK: float = 0.10
var behavior_process_interval: float = FAST_PROCESS_TICK

enum top_states {
	NONE,
	IDLE,
	ALERT,
	ALERT_MOVE,
	MOVING,
	GO_TO,
	ATTACK_SHOOT_INTENT,
	ATTACK_MELEE_INTENT,
	ATTACK_SHOOT,
	ATTACK_MELEE,
	DEAD,
	LIMBO
}
var top_state: int = top_states.IDLE
var previous_top_state: int = top_states.IDLE

const BASE_TARGET_DISTANCE_THRESHOLD: float = 150.0
var hearing_distance: float = 2000.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var melee_distance_threshold: float = 7.0
var is_never_sighted_target: bool = true
var passive_target: PhysicsBody3D
var active_target: PhysicsBody3D
var random_remembered_target_position: Vector3
var last_seen_target_position: Vector3
var strike_position: Vector3
var sound_position: Vector3
#var target_pool: Array[CharacterBody3D] = []
var target_board: Dictionary = {}
@export var groups_i_dont_like: Array[String] = [
	"players",
	"mooks",
	"punks",
	"wiseguys"
]

@export var follow_sounds: bool = false
var is_following_sound: bool = false
var sounds_heard: Array = []

var go_to_position: Vector3 = Vector3.ZERO
var last_go_to_position: Vector3 = Vector3.ZERO

@export var is_in_world_queue: bool = false
var world_queue: Node3D
var soft_process_disable: bool = false
@export var fake_process_disable: bool = false
var animation_control_pause: bool = false

# ---------------------------------------------

func _ready():
	knife_mesh.visible = false
	ouch_sounds = $Sounds.find_children("hit_*")
	die_sounds = $Sounds.find_children("die_*")
	stab_speak_sounds = $Sounds.find_children("stab_speak_*")
	welcome_speak_sounds = $Sounds.find_children("welcome_speak_*")
	knife_swipe_sounds = $Sounds.find_children("knife_swipe_*")
	
	await get_tree().create_timer(5.0).timeout
	if not active_target:
		animation_control_pause = true
	return

func _physics_process(delta):
	if fake_process_disable: return
	if is_expired: return
	
	debug_label.text = self.name + " " + top_states.keys()[top_state] + " - " + str(health)
	
	if not is_on_floor() and not soft_process_disable:
		A.apply_gravity(self, delta)
	
	
	
	if not is_dead:
		tick_control(delta)
	
	if not soft_process_disable:
		self.velocity = self.velocity.clamp(Vector3.ONE * -MAX_VELO, Vector3.ONE * MAX_VELO)
		move_and_slide()
	animation_control()
	return

func tick_control(delta):
	behavior_process_ticker += delta
	
	if behavior_process_ticker > behavior_process_interval:
		if not is_in_world_queue:
			on_behavior_process_tick()
		behavior_process_ticker = 0.0
	return

func top_state_switch_to(new_state: int):
	if previous_top_state == top_states.DEAD and previous_top_state != new_state:
		top_state = top_states.LIMBO
		prints(self.name, "Tried to change top_state out of death. Set to LIMBO until you feel like coding something to fix this.")
		return
	
	previous_top_state = top_state
	top_state = new_state
	return

func animation_control():
	#if animation_control_pause:
		#return
	if fake_process_disable: return
	
	if top_state == top_states.NONE:
		return
	
	if animation_locked:
		return
	
	if is_equipping:
		return
	
	if is_flinching:
		return
	
	if is_dead:
		A.animate(self, anim_set["die"])
		return
	
	if is_attacking and (top_state == top_states.ATTACK_SHOOT or top_state == top_states.ATTACK_SHOOT_INTENT):
		A.animate(self, "aim_shoot")
		return
	
	if is_aiming and (top_state == top_states.ATTACK_SHOOT or top_state == top_states.ATTACK_SHOOT_INTENT):
		A.animate(self, "aim_focus")
		return
	
	#if is_aiming and (top_state == top_states.ATTACK_SHOOT or top_state == top_states.ATTACK_SHOOT_INTENT):
		#A.animate(self, "cult_knife_focus")
		#return
	
	if is_attacking and (top_state == top_states.ATTACK_MELEE or top_state == top_states.ATTACK_MELEE_INTENT):
		A.animate(self, "cult_knife_stab_1")
		return
	
	if is_aiming and top_state == top_states.ATTACK_MELEE_INTENT:
		A.animate(self, "cult_knife_focus")
		return
	
	# Maybe move this up to make him run while he's moving? idk!
	if A.is_moving(self, movement_threshold):
		A.animate(self, anim_set["run"])
		return
	
	if not A.is_moving(self, movement_threshold):
		A.animate(self, anim_set["idle"])
		return
	
	
	return

func behavior_control():
	if fake_process_disable: return
	
	match top_state:
		top_states.NONE:
			return
		top_states.IDLE:
			
			hearing_control()
			
			if not active_target or active_target.is_dead:
				update_active_target()
				# Just chilling, nothing to do.
				return
			
			if active_target and get_target_distance() > target_distance_threshold:
				return
			
			if A.is_actor_viewable(self, sight_raycast, active_target) and A.is_actor_ahead_approx(self, active_target):
				if not performance_mode and is_in_world_queue:
					world_queue.remove_from_execution_group(main_group_name, self)
				
				top_state_switch_to(top_states.ALERT)
				return
			
			return
		
		top_states.ALERT:
			if is_equipping:
				return
			if is_awaiting_alert_time:
				return
			
			
			if is_never_sighted_target:
				if (U.coin_flip() + U.coin_flip()) == 2:
					U.random_choice(welcome_speak_sounds).play()
				is_never_sighted_target = false
			
			if not is_knife_equipped and not active_weapon_node and spawn_weapon != "":
				equip_weapon(spawn_weapon)
			elif not is_knife_equipped and not active_weapon_node and (spawn_weapon == "" or spawn_weapon == "knife"):
				equip_knife()
				return
			
			is_awaiting_alert_time = true
			A.face_position(self, active_target.global_position)
			await get_tree().create_timer(alert_time).timeout
			is_awaiting_alert_time = false
			# Confirm target is attackable,
			# Decide what kind of attack to use,
			# then switch to attack intent
			
			A.face_position(self, active_target.global_position)
			if not is_knife_equipped and A.is_actor_viewable(self, sight_raycast, active_target):
				top_state_switch_to(top_states.ATTACK_SHOOT_INTENT)
				return
			elif is_knife_equipped and get_target_distance() > melee_distance_threshold:
				if U.coin_flip() and not is_flinching:
					A.gradual_velo_stop(self)
				move_nav(true, active_target.global_position)
				if U.coin_flip():
					self.velocity.x += move_speed * randf_range(-1.0, 1.0)
					self.velocity.x += move_speed * randf_range(-1.0, 1.0)
			elif is_knife_equipped and get_target_distance() <= melee_distance_threshold:
				top_state_switch_to(top_states.ATTACK_MELEE_INTENT)
				return
			else:
				if $alert_fatigue.is_stopped():
					$alert_fatigue.start()
				#top_state_switch_to(top_states.ALERT) # what in the world?
				return
			
			return
		
		top_states.ALERT_MOVE:
			#Just move a little after shooting
			if is_alert_moving:
				return
			
			is_alert_moving = true
			var move_vec = Vector2(
				randi_range(-1, 1),
				randi_range(-1, 1)
			)
			if move_vec:
				var rand_speed_factor: float = randf_range(0.1, 1.0)
				A.apply_move(self, move_vec, (move_speed*rand_speed_factor))
			
			A.face_position(self, active_target.global_position)
			await get_tree().create_timer(1.0).timeout
			if not is_flinching:
				A.gradual_velo_stop(self)
			is_alert_moving = false
			top_state_switch_to(top_states.ALERT)
			return
		
		top_states.MOVING:
			return
		
		top_states.GO_TO:
			return
		
		top_states.ATTACK_SHOOT_INTENT:
			# Telegraph attack towards strike_position
			if is_aiming:
				return
			is_aiming = true
			await get_tree().create_timer(shoot_intent_time).timeout
			is_aiming = false
			top_state_switch_to(top_states.ATTACK_SHOOT)
			return
		
		top_states.ATTACK_MELEE_INTENT:
			# Telegraph attack towards strike_position
			if is_aiming:
				return
			is_aiming = true
			await get_tree().create_timer(.35).timeout
			is_aiming = false
			top_state_switch_to(top_states.ATTACK_MELEE)
			return
		
		top_states.ATTACK_SHOOT:
			if is_attacking:
				return
			is_attacking = true
			A.create_impact(self, strike_raycast, active_weapon_node.damage, active_weapon_node.knockback_force)
			active_weapon_node.primary_action(self)
			await get_tree().create_timer(.2).timeout
			if active_weapon_node and not is_dead:
				active_weapon_node.stop_action()
			is_attacking = false
			#top_state_switch_to(top_states.ALERT)
			top_state_switch_to(top_states.ALERT_MOVE)
			return
		
		top_states.ATTACK_MELEE:
			if is_attacking:
				return
			is_attacking = true
			# Strike, do damage to nearby
			U.random_choice(knife_swipe_sounds).play()
			if get_target_distance() < melee_distance_threshold and A.is_actor_ahead_approx(self, active_target):
				active_target.hit(melee_damage, U.hit_types.UNDEFINED, self, rhand_anchor.global_position)
				var chances: int = (U.coin_flip() + U.coin_flip() + U.coin_flip())
				var chance_to_voice: bool = chances == 1
				if chance_to_voice:
					U.random_choice(stab_speak_sounds).play()
			
			await get_tree().create_timer(attack_time).timeout
			is_attacking = false
			#top_state_switch_to(top_states.ALERT)
			A.face_position(self, active_target.global_position)
			top_state_switch_to(top_states.ALERT_MOVE)
			return
		
		top_states.DEAD:
			return
		
		top_states.LIMBO:
			return
		
	return

func get_target_distance():
	if not active_target:
		return -1
	return self.global_position.distance_to(active_target.global_position)

func update_active_target():
	### IF ACTIVE TARGET IS DEAD, THEN WHAT?
	# Get the closest target and set it as the active target
	target_board.clear()
	
	var target_candidates: Array = []
	for group in groups_i_dont_like:
		var group_nodes: Array = get_tree().get_nodes_in_group(group)
		if not group_nodes:
			continue
		target_candidates.append_array(group_nodes)
	
	if not target_candidates:
		return
	
	for candidate in target_candidates:
		var distance: float = self.global_position.distance_to(candidate.global_position)
		if distance > target_distance_threshold:
			continue
		target_board[candidate] = {
			"distance": distance,
			"name": candidate.name,
			"instance_id": candidate.get_instance_id()
		}
	
	if not target_board or (active_target and active_target.is_dead):
		passive_target = active_target
		active_target = null
		return
	
	var closest_target: Dictionary = {"distance": -1}
	for target in target_board.keys():
		if closest_target["distance"] < 1:
			closest_target = target_board[target]
			continue
		elif closest_target["distance"] < target_board[target]["distance"]:
			closest_target = target_board[target]
			continue
	
	active_target = instance_from_id(closest_target["instance_id"])
	animation_control_pause = false
	if logic_priority and is_in_world_queue:
		world_queue.remove_from_execution_group(main_group_name, self)
	
	if not random_remembered_target_position:
		random_remembered_target_position = active_target.global_position
	else:
		var remember_possibility: float = .5
		if randf_range(0.0, 0.99) < remember_possibility:
			random_remembered_target_position = active_target.global_position
	return

func move_nav(face_direction: bool = true, new_position: Vector3 = Vector3.ZERO, force_velo: bool = false):
	
	if new_position:
		nav_agent.target_position = new_position
	var next_position: Vector3 = nav_agent.get_next_path_position()
	if face_direction:
		A.face_position(self, next_position)
		A.apply_move(self, Vector2(0, -1), move_speed)
	elif not face_direction:
		if force_velo or not is_flinching:
			self.velocity = (next_position - self.global_position).normalized() * move_speed
	return

func equip_weapon(weapon_name: String, equip_anim: String = "draw_pistol2"):
	var weapon_item: Dictionary = Weapons.by_name(weapon_name)
	if active_weapon_node:
		active_weapon_node.free()
	
	for c in rhand_anchor.get_children():
		c.free()
	
	is_equipping = true
	A.animate(self, anim_set["equip"])
	active_weapon_name = weapon_name
	active_weapon_node = weapon_item["scene"].instantiate()
	active_weapon_node.prop_weapon = true
	rhand_anchor.add_child(active_weapon_node)
	
	active_weapon_ammo_left = weapon_item["mag_size"]
	active_weapon_mag_size = weapon_item["mag_size"]
	
	await get_tree().create_timer(equip_delay_time).timeout
	is_equipping = false
	return

func equip_knife():
	is_equipping = true
	
	A.animate(self, "draw_sleeve_knife")
	await get_tree().create_timer(0.4).timeout
	knife_mesh.visible = true
	is_knife_equipped = true
	
	is_equipping = false
	#A.animate(self, "cult_knife_idle")
	anim_set["idle"] = "cult_knife_idle"
	anim_set["run"] = "cult_knife_run"
	anim_set["die"] = "cult_die"
	return

#func flinch_control(hitbox: CollisionShape3D):
func flinch_control():
	is_flinching = true
	
	#var hitbox_group: String = ""
	#if hitbox.name == "hitbox_head":
		#hitbox_group = "head"
	#elif hitbox.name == "hitbox_chest":
		#hitbox_group = "chest"
	#elif hitbox.name.contains("_upper_leg_"):
		#hitbox_group = "legs"
	#elif hitbox.name.contains("_lower_leg_"):
		#hitbox_group = "knees"
	#elif hitbox.name.contains("_foot_"):
		#hitbox_group = "knees"
	#
	#var flinch_anim: String
	#match hitbox_group:
		#"head":
			#flinch_anim = "flinch_head"
		#"chest":
			#flinch_anim = ["flinch_torso1", "flinch_torso2"][randi_range(0, 1)]
		#"hitbox_lower":
			#flinch_anim = "flinch_lower1"
		#"hitbox_knees":
			#flinch_anim = "flinch_lower1"
		#_:
			#flinch_anim = "flinch"
	var flinch_anim: String = ""
	if is_knife_equipped:
		flinch_anim = "cult_knife_flinch"
	else:
		flinch_anim = "flinch_torso2"
	A.animate(self, flinch_anim)
	
	await get_tree().create_timer(1.0).timeout
	is_flinching = false
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	#if is_dead: return # If no dead flinch, uncomment
	#if is_dead:
		#prints(self.name, anim.assigned_animation, anim.current_animation)
		#if anim.assigned_animation != anim_set["die"]:
			#A.animate(self, anim_set["die"])
	
	var adjusted_damage: int = damage
	
	var closest_hitbox: CollisionShape3D
	if not hit_pos and health > 0:
		flinch_control()
	if hit_pos and health > 0:
		var hitbox_positions: Dictionary = {}
		for hitbox in hitboxes:
			hitbox_positions[hitbox.global_position] = hitbox
		
		var closest_position: Vector3 = U.get_closest_position(hit_pos, hitbox_positions.keys())
		closest_hitbox = hitbox_positions[closest_position]
		#flinch_control(closest_hitbox)
		flinch_control()
	
	if closest_hitbox and closest_hitbox.name.contains("head"):
		adjusted_damage *= 3
	
	health -= adjusted_damage
	if health > 0:
		U.random_choice(ouch_sounds).play()
	
	if health > 0 and health <= original_health and U.coin_flip():
		var fast_speed: float = move_speed * 1.6
		var slow_speed: float = move_speed * 0.5
		#move_speed = fast_speed if U.coin_flip() else slow_speed
		move_speed = fast_speed
		await get_tree().create_timer(1.5).timeout
		get_tree().create_tween().tween_property(self, "move_speed", 6, 5.0)
		if U.coin_flip():
			top_state_switch_to(top_states.ALERT_MOVE)
	
	if caller and not is_dead:
		var caller_distance: float = self.global_position.distance_to(caller.global_position)
		if caller_distance > target_distance_threshold:
			target_distance_threshold = caller_distance
		active_target = caller
		A.face_position(self, active_target.global_position)
	
	if health < 1 and not is_dead:
		kill_actor()
	return

func kill_actor():
	is_dead = true
	animation_control_pause = false
	
	U.random_choice(die_sounds).play()
	
	if is_in_world_queue:
		world_queue.remove_from_execution_group(main_group_name, self)
	
	if not self.velocity.is_zero_approx():
		var tween: Tween = get_tree().create_tween()
		tween.tween_property(self, "velocity", Vector3.ZERO, 1.0)
	
	for group in ["actors"]:
		for p in get_tree().get_nodes_in_group(group):
			self.add_collision_exception_with(p)
	
	self.remove_from_group("hit_takers")
	$CollisionShape3D.disabled = true
	$dead_colshape.disabled = false
	
	if active_weapon_node and not is_knife_equipped:
		drop_weapon()
	
	if drop_on_death:
		drop_spawnable(drop_on_death)
	
	alert_other_group_members()
	await get_tree().create_timer(5.0).timeout
	soft_process_disable = true
	if anim.assigned_animation != anim_set["die"]:
		A.animate(self, anim_set["die"])
	#get_tree().create_tween().tween_callback(func(): soft_process_disable = true).set_delay(3.0)
	
	expire_actor()
	return

func expire_actor():
	for hitbox in hitboxes:
		hitbox.disabled = true
	self.process_mode = Node.PROCESS_MODE_DISABLED
	is_expired = true
	return

func drop_weapon():
	var world_item: RigidBody3D = Weapons.by_name(spawn_weapon)["world_item"].instantiate()
	Blackboard.current_world.add_child(world_item)
	world_item.global_position = rhand_anchor.global_position
	world_item.rotation_degrees.y = 180
	
	world_item.add_collision_exception_with(self)
	for p in get_tree().get_nodes_in_group("players"):
		world_item.add_collision_exception_with(p) # This shouldn't go here!
	
	world_item.linear_velocity = Vector3(0, 10, 0)
	world_item.angular_velocity = Vector3(0, 0, 5)
	
	for c in rhand_anchor.get_children():
		c.queue_free()
	
	return

func drop_spawnable(world_item_scene: PackedScene):
	#var world_item: RigidBody3D = Weapons.by_name(spawn_weapon)["world_item"].instantiate()
	var spawn_position: Vector3 = $mid_anchor.global_position
	#await get_tree().create_timer(1.8).timeout
	await get_tree().create_timer(0.5).timeout
	var world_item: Node3D = world_item_scene.instantiate()
	Blackboard.current_world.add_child(world_item)
	world_item.global_position = spawn_position
	world_item.rotation_degrees.y = 180
	
	#world_item.add_collision_exception_with(self)
	#for p in get_tree().get_nodes_in_group("players"):
		#world_item.add_collision_exception_with(p) # This shouldn't go here!
	
	if is_instance_of(world_item, RigidBody3D):
		world_item.linear_velocity = Vector3(0, 10, 0)
		world_item.angular_velocity = Vector3(0, 0, 5)
	
	#for c in rhand_anchor.get_children():
		#c.queue_free()
	
	return

func on_behavior_process_tick():
	#if is_dead and top_state == top_states.DEAD:
	if is_dead:
		return
	
	behavior_control()
	return


func _on_alert_fatigue_timeout():
	if top_state == top_states.ALERT:
		top_state_switch_to(top_states.IDLE)
	return

func _world_queue_execute():
	if is_dead:
		return
	
	if not active_target and is_on_floor():
		update_active_target()
		if not active_target:
			soft_process_disable = true
			return
		else:
			soft_process_disable = false
	
	on_behavior_process_tick()
	return

func _on_perf_timver_timeout():
	if is_dead:
		$perf_timer.stop()
	return

func _weapon_action_triggered(caller: Node3D):
	A.create_impact(self, strike_raycast, active_weapon_node.damage, active_weapon_node.knockback_force)
	return

func hearing_control():
	sounds_heard = A.check_for_sounds(self, hearing_distance)
	for sound in sounds_heard:
		var sound_parent: Node3D = sound.get_parent()
		#if sound_parent in target_board:
		if sound_parent.is_in_group("players"):
			A.face_position(self, sound_parent.global_position)
		#if enable_sound_following:
			#sound_position = sound.global_position
			#is_following_sound = true
			#alert_fatigue.start()
			break
	return

func alert_other_group_members():
	if not active_target:
		return
	if not active_target.is_in_group("actors"):
		return
	for group_member in get_tree().get_nodes_in_group(main_group_name):
		if group_member == self:
			continue
		var distance_to_buddy: float = self.global_position.distance_to(group_member.global_position)
		if distance_to_buddy > target_distance_threshold:
			continue
		group_member.active_target = self.active_target
		#group_member.update_active_target()
		if U.coin_flip():
			A.face_position(group_member, self.active_target.global_position)
	return

# RQRR
