extends CharacterBody3D

@onready var debug_label: Label3D = $debug_label

@export var health: int = 130
@export var override_skill_values: bool = false
@export var throw_wait_time: float = 1.0
@export var intent_wait_time: float = 1.0
@export var attack_delay_time: float = 0.8
const SPEED = 5.0
const JUMP_VELOCITY = 4.5
@onready var anim: AnimationPlayer = $model/AnimationPlayer
@onready var model: Node3D = $model
var turn_speed: float = 0.50
var move_speed: float = 5.0
var last_anim: String
var queued_anims: Array[String] = []

# Get the gravity from the project settings to be synced with RigidBody nodes.
var gravity = (ProjectSettings.get_setting("physics/3d/default_gravity") * 6)

@onready var eyes: Node3D = $mid_anchor/eyes
@onready var base_eyes: Node3D = $base_anchor/eyes
@onready var sight_raycast: RayCast3D = $sight_raycast
@onready var rhand_anchor: Node3D = $model/Armature/Skeleton3D/rhand_attachment/rhand_anchor
@onready var lhand_anchor: Node3D = $model/Armature/Skeleton3D/lhand_attachment/lhand_anchor
#var held_knife: Resource = preload("res://Scenes/fat_held_butcher_knife.tscn")
@onready var held_knife: Node3D = $model/Armature/Skeleton3D/rhand_attachment/rhand_anchor/fat_held_butcherknife
var flying_knife: Resource = preload("res://Scenes/flying_butcher_knife.tscn")

@onready var hitboxes: Array[CollisionShape3D] = [
	$hitbox_head,
	$hitbox_chest,
	$hitbox_lower,
	$hitbox_foot
]

var anim_set: Dictionary = {
	"idle": "unarmed_idle",
	"throw": "throw_overhand"
}

##############################################################################

var behavior_process_ticker: float = 0.0
@export var behavior_process_interval: float = 0.1

enum top_states {
	NONE,
	IDLE,
	THROW_INTENT,
	THROW_ATTACK,
	ALERT,
	MOVING,
	DEAD
}

@export var top_state: int = top_states.IDLE
var is_dead: bool = false

@export var groups_i_dont_like: Array[String] = [
	"players",
	"player_minions",
	"punks"
]

# Make a group hostility mode too!
# - mode_1: Hostile towards only groups I don't like
# - mode_2: Hostile towards all groups except the ones I like

@onready var attack_delay: Timer = $attack_delay
@onready var intent_wait: Timer = $intent_wait
@onready var throw_timer: Timer = $throw_timer
@onready var flinch_timer: Timer = $flinch_timer
@onready var move_timer: Timer = $move_timer
var target_distance_threshold: float = 70.0
var passive_target: PhysicsBody3D
#var active_target: CharacterBody3D
var active_target: PhysicsBody3D
var target_pool: Array[CharacterBody3D] = []
var strike_position: Vector3

##############################################################################
var is_flinching: bool = false


func _ready():


	match Global.skill_level:
		Global.skill_levels.SWEET_DREAMS:
			attack_delay.wait_time = 1.5
			intent_wait.wait_time = 1.0
			throw_timer.wait_time = 1.0
		Global.skill_levels.ROUGH_NIGHT:
			attack_delay.wait_time = 1.0
			intent_wait.wait_time = .8
			throw_timer.wait_time = .8
		Global.skill_levels.NIGHT_TERROR:
			attack_delay.wait_time = .5
			intent_wait.wait_time = .5
			throw_timer.wait_time = .5
		Global.skill_levels.PARALYSIS:
			attack_delay.wait_time = .1
			intent_wait.wait_time = .1
			throw_timer.wait_time = .1
	
	if override_skill_values:
		attack_delay.wait_time = attack_delay_time
		intent_wait.wait_time = intent_wait_time
		throw_timer.wait_time = throw_wait_time
	return

func _physics_process(delta):
	if not is_dead:
		tick_control(delta)

	if not is_on_floor():
		apply_gravity(delta)

	#behavior_control() # Moved (temporarily?) to on_behavior_process_tick()

	move_and_slide()
	return

func tick_control(delta):
	behavior_process_ticker += delta
	if behavior_process_ticker > behavior_process_interval:
		on_behavior_process_tick()
		behavior_process_ticker = 0.0
	return


func apply_gravity(delta):
	velocity.y -= gravity * delta
	return

func apply_move(input_dir: Vector2):
	# Get the input direction and handle the movement/deceleration.
	# As good practice, you should replace UI actions with custom gameplay actions.
	# var input_dir = Input.get_vector("ui_left", "ui_right", "ui_up", "ui_down")
	var direction = (transform.basis * Vector3(input_dir.x, 0, input_dir.y)).normalized()
	if direction:
		velocity.x = direction.x * SPEED
		velocity.z = direction.z * SPEED
	else:
		velocity.x = move_toward(velocity.x, 0, SPEED)
		velocity.z = move_toward(velocity.z, 0, SPEED)
	return

func apply_jump(floor_check: bool = true):
	if floor_check and not is_on_floor(): return
	velocity.y = JUMP_VELOCITY
	return

func face_position(raw_position: Vector3 = Vector3.ZERO, parallel: bool = false):
	var target_position = raw_position
	if parallel:
		target_position.y = self.global_position.y
	# This could probably be done with tween, without the eyes node
	eyes.look_at(target_position, Vector3.UP)
	#model.rotation.x = (eyes.rotation.x * .9)
	#model.rotation = eyes.rotation
	# Using the model position is probably a bad idea because the physics will not have been rotated
	self.global_rotation.y = eyes.global_rotation.y
	return

func look_towards(target_position: Vector3, turn_amount: float, flip_forward: bool):
	# This could probably be done with tween, without the eyes node
	eyes.look_at(target_position, Vector3.UP, true)
	#model.rotation.x = (eyes.rotation.x * .9)
	model.rotate_y(eyes.rotation.y * turn_amount)
	if flip_forward:
		model.rotation.y += 180
	return

func look_towards_parallel(target_position: Vector3, turn_amount: float):
	var adjusted_target: Vector3 = target_position
	adjusted_target.y = self.global_position.y
	look_towards(adjusted_target, turn_amount, true)
	return

func is_body_viewable(target_body: PhysicsBody3D):
	# var original_global_rotation: Vector3 = sight_raycast.global_rotation
	# May be an issue later not resetting the raycast orientation but whatever for now!
	if not target_body:
		return false
	
	sight_raycast.look_at(target_body.global_position, Vector3.UP)
	var collider = sight_raycast.get_collider()
	return collider == target_body

func anim_h(anim_name: String, is_queued: bool = true, force_anim: bool = false):
	if anim_name == last_anim and not force_anim:
		return
	anim.play(anim_name)
	if is_queued:
		queued_anims.push_back(anim_name)
	last_anim = anim_name
	return

func get_target_candidates(group_list: Array[String]):
	var candidates: Array[CharacterBody3D] = []
	for group in group_list:
		for actor in get_tree().get_nodes_in_group(group):
			candidates.push_front(actor)
	return candidates

func update_targets_by_distance():
	if self.name == "fat_wiseguy5":
		pass
	var target_candidates: Array = get_target_candidates(groups_i_dont_like)
	var lowest_distance: float = -1.0
	var closest_target: CharacterBody3D
	if not target_candidates:
		return
	
	for target in target_candidates:
		var target_distance: float = self.global_position.distance_to(target.global_position)
		if target_distance < target_distance_threshold:
			if not target in target_pool:
				target_pool.push_front(target)
			if lowest_distance < 0.0:
				lowest_distance = target_distance
				closest_target = target
				continue
			elif lowest_distance > 0.0 and target_distance < lowest_distance:
				lowest_distance = target_distance
				closest_target = target
				continue
			elif lowest_distance > 0.0 and target_distance > lowest_distance:
				continue
	
	if closest_target:
		active_target = closest_target
	elif not closest_target:
		passive_target = active_target
		active_target = null
	return

func is_target_valid(target: CharacterBody3D):
	if not active_target and not passive_target:
		return false

	var conditions: Array = [
		(active_target != null),
		is_body_viewable(active_target),
	]
	
	var conditions_met: bool = false
	for condition in conditions:
		if condition == true:
			conditions_met = true
			continue
		elif condition == false:
			conditions_met = false
			break
			
	return conditions_met

func top_state_switch_to(new_state: int):
	#var previous_state = top_state
	top_state = new_state
	return

func on_behavior_process_tick():
	behavior_control()
	return

func spawn_throwing_knife(target_pos: Vector3):
	var new_knife: RigidBody3D = flying_knife.instantiate()
	
	Blackboard.current_world.add_child(new_knife)
	new_knife.global_position = rhand_anchor.global_position
	new_knife.global_position.y += 2.0
	new_knife.add_collision_exception_with(self)
	new_knife.thrower = self
	
	var knife_speed: float = 35
	base_eyes.look_at(target_pos, Vector3.UP)
	new_knife.global_transform.basis = base_eyes.global_transform.basis
	new_knife.lock_rotation = true
	var adjusted_target_pos = target_pos
	adjusted_target_pos.y += 1.5
	var distance_scale: float = 0.9
	var target_distance: float = new_knife.global_position.distance_to(adjusted_target_pos)
	var distance_force: float = distance_scale * target_distance
	var direction: Vector3 = (adjusted_target_pos - new_knife.global_position).normalized() * (knife_speed + distance_force)
	new_knife.apply_impulse(direction, new_knife.global_position)
	return


func behavior_control():
	match top_state:
		top_states.NONE:
			return
		top_states.IDLE:
			update_targets_by_distance()
			#held_knife.visible = false
			if is_target_valid(active_target):
				top_state_switch_to(top_states.THROW_INTENT)
			else:
				anim_h(anim_set["idle"])
		
		top_states.THROW_INTENT:
			if not attack_delay.is_stopped():
				return
		# A lazy way to improve experience with very little work:
		# make a "look around" animation to play while the intent wait timer is running.
		# it can even be like, looking towards what would be the FOV instead of all around (which might look silly)
			
			update_targets_by_distance()
			
			if Global.skill_level == Global.skill_levels.PARALYSIS:
				var random_walk_chance: int = randi_range(0, 3)
				if random_walk_chance == 2:
					top_state_switch_to(top_states.MOVING)
					move_timer.start()
					return
			
			var vtarg: bool = is_target_valid(active_target)
			if vtarg:
				held_knife.visible = true
				strike_position = active_target.global_position
				#face_position(active_target.global_position, false)
				face_position(strike_position, false)
				anim_h("throw_prime")
			else:
				anim_h("armed_idle")
			
			if vtarg:
				intent_wait.stop()
				attack_delay.start()
				#top_state_switch_to(top_states.THROW_ATTACK)
				return
			if not vtarg and intent_wait.is_stopped():
				intent_wait.start()
				return
			elif not vtarg and not intent_wait.is_stopped():
				return
			#elif vtarg and not intent_wait.is_stopped():
			#	intent_wait.stop()
			
			#face_position(active_target.global_position, false)
			anim_h("armed_idle")
			if not active_target:
				top_state_switch_to(top_states.IDLE) # Should be ALERT
			

			# 1. Wait and act to meet conditions allowing for a knife throw
			# 2a. If conditions met, change state and throw knife
			# 2b. If conditions not met in limits (e.g. time limit), change state
		
		top_states.THROW_ATTACK:
			# 1. Throw the knife at the target
			# 2. Determine if another knife should be thrown immediately, if not, change state
			if not throw_timer.is_stopped():
				return
			
			# Last second aim at you part vvv
			var throwing_target: CharacterBody3D
			if active_target:
				throwing_target = active_target
			elif not active_target and passive_target:
				throwing_target = passive_target
			elif not active_target and not passive_target:
				return
			# Last second aim at you part ^^^

			if not active_target and not passive_target:
				return
			
			# Last second aim at you part vvv
			face_position(throwing_target.global_position)
			anim_h(anim_set["throw"], false, true)
			spawn_throwing_knife(throwing_target.global_position)
			# Last second aim at you part ^^^
			
			#spawn_throwing_knife(strike_position)
			throw_timer.start()
		
		top_states.ALERT:
			# Target was just seen but cannot engage (out of range, not viewable, etc.)
			# Be ready in case they reappear until limit is reached (e.g. timelimit)
			pass
		top_states.MOVING:
			if not active_target:
				prints(self.name, "No active target in state MOVING, may loop.")
				return
			A.face_position(self, active_target.global_position)
			A.apply_move(self, Vector2.UP, move_speed)
			anim_h("throw_prime_walk")
			return
		
		top_states.DEAD:
			pass
	return



func nnyeah():
	var random_timer_time: float = randf_range(0.8, 3.8)
	var random_nnyeah_pitch: float = randf_range(.85, 1.1)
	$nnyeah.pitch_scale = random_nnyeah_pitch
	$nnyeah_timer.wait_time = random_timer_time
	$nnyeah_timer.start()
	return




func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	#if is_dead: return
	
	var closest_hitbox: CollisionShape3D
	if hit_pos:
		var hitbox_positions: Dictionary = {}
		for hitbox in hitboxes:
			hitbox_positions[hitbox.global_position] = hitbox
		
		var closest_position: Vector3 = U.get_closest_position(hit_pos, hitbox_positions.keys())
		closest_hitbox = hitbox_positions[closest_position]
	
	if caller:
		active_target = caller
	
#	if hit_type == U.hit_types.UNDEFINED:
	if is_dead and flinch_timer.is_stopped():
		anim_h("dead_flinch", false, true)
		return
	
	attack_delay.stop() # EXPERIMENTAL
	
	if closest_hitbox == $hitbox_head:
		health -= damage * 3
	else:
		health -= damage
	
	if health > 0:
		flinch_timer.start()
		is_flinching = true
		anim_h("flinch", false, true)
		$Sounds/wiseguy_ouch1.play()
	elif health < 1:
		kill_actor()
	return

func kill_actor():
	top_state_switch_to(top_states.DEAD)
	anim_h("die1", false, true)
	is_dead = true
	
	self.remove_from_group("hit_takers")
	
	for group in ["players", "player_minions"]:
		for p in get_tree().get_nodes_in_group(group):
			self.add_collision_exception_with(p)
	return

func _on_nnyeah_timer_timeout():
	$nnyeah.play()
	return

func _on_debug_timer_timeout():
	#debug_label.text = top_states.keys()[top_state]
	return

func _on_intent_wait_timeout():
	if is_dead: return
	top_state_switch_to(top_states.IDLE)
	return


func _on_throw_timer_timeout():
	if is_dead: return
	top_state_switch_to(top_states.THROW_INTENT)
	return

func _on_flinch_timer_timeout():
	is_flinching = false
	return


func _on_attack_delay_timeout():
	top_state_switch_to(top_states.THROW_ATTACK)
	return


func _on_move_timer_timeout():
	top_state_switch_to(top_states.THROW_ATTACK)
	return
