extends CharacterBody3D

signal speech_used(speech_name)

const JUMP_VELOCITY = 4.5
const BASE_MOVE_SPEED = 10.0
const MAX_VELO: float = 21.0
var move_speed: float = BASE_MOVE_SPEED
var movement_threshold: float = 9.0
var run_scale = 1.2

@onready var eyes: Node3D = self.get_node("mid_anchor/eyes")
@onready var debug_label: Label3D = self.get_node("debug_label")
@onready var anim: AnimationPlayer = self.get_node("model/AnimationPlayer")
@onready var rhand_anchor: Node3D = self.get_node("model/Armature/Skeleton3D/attachment_hand_R/anchor")
@onready var sight_raycast: RayCast3D = $sight_raycast
@onready var strike_raycast: RayCast3D = $strike_raycast
@onready var nav_agent: NavigationAgent3D = self.get_node("NavigationAgent3D")
@onready var right_feeler: RayCast3D = $base_anchor/right_feeler
@onready var left_feeler: RayCast3D = $base_anchor/left_feeler
@onready var alert_fatigue: Timer = $alert_fatigue
var sound_melee_swipe: AudioStreamPlayer3D
var sound_melee_hit: AudioStreamPlayer3D
@onready var sound_throw_land: AudioStreamPlayer3D = $Sounds/throw_land

@export var health: int = 120 # 55
var starting_health: int
@export var spawn_dancing: bool = false
@export var spawn_weapon: String
@export var use_knife: bool = true
@export var force_random_melee_choice: bool = false
@export var melee_damage: float = 16.0
@export var melee_recoil: float = -30.0
@export var melee_intent_time: float = 1.0
var punch_intent_time: float = melee_intent_time * .25
@export var melee_cooloff: float = 0.5
@export var look_around_cooloff: float = 0.8
@export var extended_attack_cooloff: float = 0.9
@export var speak_cooloff: float = 2.0
@export var shirt: String
@export var pant: String
@export var hair: String
var is_dead: bool = false

var active_weapon_name: String
var active_weapon_node: Node3D
var active_weapon_ammo_left: int
var active_weapon_mag_size: int
var is_wandering: bool = false
var is_crazy: bool = true
var is_fastboy: bool = false
var is_equipping: bool = false
var is_aiming: bool = false
var is_attacking: bool = false
var is_melee_attacking: bool = false
var is_recently_attacking: bool = false
var is_recently_turned: bool = false
var is_recently_speaking: bool = false
var is_attack_ready: bool = true
var is_flinching: bool = false
var is_being_carried: bool = false
var is_being_thrown: bool = false
var animation_locked: bool = false

var anim_set: Dictionary = {
	"idle": "unarmed_idle",
	"run": "angry_run",
	"melee_intent": "",
	"melee_attack": "",
	"flinch": "flinch",
	"die": "die_1"
}
var external_anims: Dictionary = {
	"being_carried": preload("res://Staging/Models/Animations/lifter_being_carried.res")
}

var voice_base_words: Array[String] = [
	"get_out_of_here"
]


var ouch_sounds: Array[AudioStreamPlayer3D] = []
var die_sounds: Array[AudioStreamPlayer3D] = []
@onready var crazy_sounds_container: Node3D = $Sounds/voice/Crazy
@onready var normal_sounds_container: Node3D = $Sounds/voice/Normal

var behavior_process_ticker: float = 0.0
const FAST_PROCESS_TICK: float = 0.049
const DEFAULT_PROCESS_TICK: float = 0.10
@export var behavior_process_interval: float = FAST_PROCESS_TICK

var wander_possibility: float = 0.3

enum top_states {
	NONE,
	DANCING,
	IDLE,
	ALERT,
	MOVING,
	LOW_HEALTH_ALERT,
	GO_TO,
	ATTACK_SHOOT_INTENT,
	ATTACK_MELEE_INTENT,
	ATTACK_SHOOT,
	ATTACK_MELEE,
	DEAD
}
var top_state: int = top_states.IDLE
var previous_top_state: int = top_states.IDLE

enum crazy_states {
	NONE,
	CRAZY,
	UNHINGED
}
var crazy_state: int = crazy_states.NONE

const BASE_TARGET_DISTANCE_THRESHOLD: float = 25.0
@export var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var melee_distance_threshold: float = 7.0
var passive_target: CharacterBody3D
#var active_target: CharacterBody3D
var active_target: PhysicsBody3D
var random_remembered_target_position: Vector3
var last_seen_target_position: Vector3
var pre_carried_rotation: Vector3
var strike_position: Vector3
var sound_position: Vector3
#var target_pool: Array = []
var target_board: Dictionary = {}
@export var groups_i_dont_like: Array[String] = ["players"]

@export var follow_sounds: bool = false
var is_following_sound: bool = false
var sounds_heard: Array = []

var go_to_position: Vector3 = Vector3.ZERO
var last_go_to_position: Vector3 = Vector3.ZERO

# ---------------------------------------------

func _ready():
	random_remembered_target_position = self.global_position
	starting_health = health
	$model/Armature/Skeleton3D/knife.visible = false
	
	if spawn_dancing:
		anim_set["idle"] = ["gesture_guy_dance", "gesture_punk_dance_1"].pick_random()
	
	alert_fatigue.connect("timeout", _on_alert_fatigue_timeout)
	for sound in $Sounds/voice/Extra.find_children("die_*"):
		die_sounds.append(sound)
	
	if not "recent_punk_sounds" in Blackboard.dump_table:
		Blackboard.dump_table["recent_punk_sounds"] = []
	
	self.connect("speech_used", _speech_used)
	
	if shirt.contains("random"):
		var shirts: Array = $model/Armature/Skeleton3D.find_children("*shirt*")
		for shirt in shirts:
			shirt.visible = false
		U.random_choice(shirts).visible = true
	
	if pant.contains("random"):
		var pants: Array = $model/Armature/Skeleton3D.find_children("*pant*")
		for pant in pants:
			pant.visible = false
		U.random_choice(pants).visible = true
	
	if hair.contains("random"):
		var hairs: Array = $model/Armature/Skeleton3D.find_children("*hair*")
		for hair in hairs:
			hair.visible = false
		U.random_choice(hairs).visible = true
	
	if force_random_melee_choice:
		use_knife = true if randi_range(0, 1) == 1 else false
	
	if use_knife:
		anim_set["melee_intent"] = "stab_intent_1"
		anim_set["melee_attack"] = "stab_fire_1"
		melee_damage = 20.0
		sound_melee_swipe = $Sounds/stab_swipe
		sound_melee_hit = $Sounds/stab_hit
	else:
		anim_set["melee_intent"] = "punch_intent_1"
		anim_set["melee_attack"] = "punch_fire_1"
		melee_damage = 10.0
		sound_melee_swipe = $Sounds/punch_swipe
		sound_melee_hit = $Sounds/punch_hit
	
	return

func _physics_process(delta):
	if debug_label.visible:
		debug_label.text = self.name + " " + top_states.keys()[top_state] + " - " + str(anim.assigned_animation)
		$debug_label2.text = str(self.health)
	
	
	if is_being_thrown and is_on_floor() and not sound_throw_land.playing:
		sound_throw_land.play()
	
	if is_being_carried:
		animation_control()
		return
	
	if not is_on_floor():
		A.apply_gravity(self, delta)
	
	if not is_dead:
		tick_control(delta)
	
	animation_control()
	self.velocity = clamp(self.velocity, -Vector3.INF, Vector3(MAX_VELO, 1.0, MAX_VELO))
	move_and_slide()
	return

func tick_control(delta):
	behavior_process_ticker += delta
	if behavior_process_ticker > behavior_process_interval:
		sound_listening_control()
		on_behavior_process_tick()
		behavior_process_ticker = 0.0
	return

func top_state_switch_to(new_state: int):
	if not alert_fatigue.is_stopped():
		alert_fatigue.stop()
	
	previous_top_state = top_state
	top_state = new_state
	return

func sound_listening_control():
	sounds_heard.clear()
	var world_sounds: Array = get_tree().get_nodes_in_group("world_sounds")
	for sound in world_sounds:
		var is_in_hearing_range: bool
		if not sound.sound_distance:
			is_in_hearing_range = true
		if sound.sound_distance and self.global_position.distance_to(sound.global_position) < sound.sound_distance:
			is_in_hearing_range = true
		if not is_in_hearing_range:
			continue
		sounds_heard.append(sound)
	
	for sound in sounds_heard:
		if "gunshot" in sound.tags:
			is_fastboy = true
			anim_set["run"] = "sprint"
			move_speed = BASE_MOVE_SPEED * 2.5
	return

func animation_control():
	if is_being_carried:
		A.animate(self, "lifter_being_carried", true)
		return
	
	if top_state == top_states.DANCING:
		return
	
	if top_state == top_states.NONE:
		return
	
	if animation_locked:
		return
	
	
	if is_dead:
		A.animate(self, U.random_choice(["die_1", "die_2"]))
		#A.animate(self, anim_set["die"])
		animation_locked = true
		return
	
	#if top_state == top_states.ATTACK_MELEE and is_melee_attacking:
	if is_melee_attacking and top_state == top_states.ATTACK_MELEE:
		A.animate(self, anim_set["melee_attack"], true)
		return
	
	if is_flinching:
		A.animate(self, anim_set["flinch"])
		return
	
	if is_recently_turned:
		A.animate(self, "look_around")
		return
	
	if is_recently_attacking:
		return
	
	
	if top_state == top_states.ATTACK_MELEE_INTENT and is_melee_attacking:
		A.animate(self, anim_set["melee_intent"])
		return
	
	
	if not A.is_moving(self, movement_threshold):
		A.animate(self, anim_set["idle"])
		return
	
	if A.is_moving(self, movement_threshold) and is_recently_speaking and is_crazy:
		var run_anim: String
		var run_anims: Array[String] = ["angry_run", "crazy_run", "unhinged_run"]
		if is_fastboy:
			run_anim = "sprint"
		else:
			run_anim = run_anims[crazy_state]
		A.animate(self, run_anim)
		return
	
	if A.is_moving(self, movement_threshold):
		A.animate(self, anim_set["run"])
		return
	return

func behavior_control():
	match top_state:
		top_states.NONE:
			return
		
		
		top_states.IDLE:
			#update_active_target()
			
			if not active_target:
				update_active_target()
				# Just chilling, nothing to do.
				return
			
			if active_target and get_target_distance() > target_distance_threshold:
				return
			
			if A.is_actor_viewable(self, sight_raycast, active_target):
				if randi_range(0,3) == 2 or A.is_actor_ahead_approx(self, active_target):
					#if spawn_dancing:
						#anim_set["idle"] = "unarmed_idle"
					$model/Armature/Skeleton3D/knife.visible = true if use_knife else false
					speak("i_see_you", 0.1)
					top_state_switch_to(top_states.ALERT)
				return
			return
		
		top_states.ALERT:
			if not is_dead and health < starting_health:
				top_state_switch_to(top_states.LOW_HEALTH_ALERT)
			if not active_target: #  and not passive_target:
				top_state_switch_to(top_states.IDLE)
				return
			
			if active_target and active_target.is_dead:
				active_target = null
				passive_target = null
				top_state_switch_to(top_states.IDLE)
				return
			
			#if active_target and health < starting_health and not is_fastboy:
				#if randf_range(0.01, 0.99) < wander_possibility:
					#nav_agent.target_position = random_remembered_target_position
					#is_wandering = true
					#
					#top_state_switch_to(top_states.MOVING)
					#A.apply_move(self, Vector2.UP, move_speed)
					#await get_tree().create_timer(2.8).timeout
					#top_state_switch_to(top_states.ALERT)
					#is_wandering = false
					#wander_possibility = 0.0
					#return
			
			alert_fatigue.start()
			
			
			var y_look_offset: float = 4.0 if not active_target.is_in_group("players") else 0.0
			var is_target_attackable: bool = (
				active_target and \
				A.is_actor_viewable(self, sight_raycast, active_target, y_look_offset) and \
				A.is_actor_ahead_approx(self, active_target)
			)
			
			var debug_is_actor_viewable: bool = A.is_actor_viewable(self, sight_raycast, active_target, y_look_offset)
			var debug_is_actor_ahead: bool = A.is_actor_ahead_approx(self, active_target)
			
			if self.name == "punk4" and health < starting_health and A.is_actor_ahead_approx(self, active_target):
				pass
			#if active_target and is_recently_attacking and not A.is_actor_ahead_approx(self, active_target):
			if active_target and not A.is_actor_ahead_approx(self, active_target):
				#if not is_recently_turned:
				if not is_recently_turned:
					speak("get", 0.01)
					gradual_velo_stop()
					if not is_being_thrown:
						random_turn(135)
						is_recently_turned = true
						await get_tree().create_timer(look_around_cooloff).timeout
						is_recently_turned = false
				return
			
			if not is_target_attackable:
				return
			
			if is_target_attackable and active_weapon_node:
				top_state_switch_to(top_states.ATTACK_SHOOT_INTENT)
				return
			
			if is_target_attackable and not active_weapon_node:
				var target_distance: float = self.global_position.distance_to(active_target.global_position)
				if target_distance < melee_distance_threshold:
					top_state_switch_to(top_states.ATTACK_MELEE_INTENT)
				else:
					var thing_to_say: String = U.random_choice(
						[
							"get_out",
							"cmere",
							"come_here",
							"gonna_hurt",
							"bleed"
						]
					)
					speak(thing_to_say, 0.01)
					move_nav(true, active_target.global_position)
					#if not is_fastboy:
						#move_feeler_control(true)
				return
			return
		
		top_states.MOVING:
			if is_wandering:
				return
			
			if self.global_position.distance_to(nav_agent.target_position) > 23.0:
				move_nav(true)
			else:
				top_state_switch_to(top_states.ALERT)
			#elif active_target:
				#gradual_velo_stop()
				#A.face_position(self, active_target.global_position)
			#else:
				#random_turn()
			return
		
		top_states.LOW_HEALTH_ALERT:
			if not active_target: #  and not passive_target:
				top_state_switch_to(top_states.IDLE)
				return
			
			if active_target and active_target.is_dead:
				active_target = null
				passive_target = null
				top_state_switch_to(top_states.IDLE)
				return

			var y_look_offset: float = 4.0 if not active_target.is_in_group("players") else 0.0
			var is_target_attackable: bool = (
				active_target and \
				A.is_actor_viewable(self, sight_raycast, active_target, y_look_offset) and \
				A.is_actor_ahead_approx(self, active_target)
			)
			
			if not is_target_attackable:
				$debug_label.text = self.name + " target is not attackable."
				A.face_position(self, active_target.global_position)
				return
			
			if is_target_attackable and active_weapon_node:
				top_state_switch_to(top_states.ATTACK_SHOOT_INTENT)
				return
			
			if is_target_attackable and not active_weapon_node:
				var target_distance: float = self.global_position.distance_to(active_target.global_position)
				if target_distance < melee_distance_threshold:
					top_state_switch_to(top_states.ATTACK_MELEE_INTENT)
				else:
					var thing_to_say: String = U.random_choice(
						[
							"get_out",
							"cmere",
							"come_here",
							"gonna_hurt",
							"bleed"
						]
					)
					speak(thing_to_say, 0.01)
					move_nav(true, active_target.global_position)
					if not is_fastboy:
						move_feeler_control(true)
				return
			return
		
		top_states.GO_TO:
			return
		
		top_states.ATTACK_SHOOT_INTENT:
			return
		
		top_states.ATTACK_MELEE_INTENT:
			if is_melee_attacking:
				return
			
			gradual_velo_stop()
			
			if health < (starting_health * .40) and not is_melee_attacking:
				var move_back_chance: float = 0.50
				if randf_range(0.0, 0.99) < move_back_chance:
					A.apply_move(self, Vector2.DOWN, move_speed)
					#return
			
			is_melee_attacking = true
			
			var delay: float = melee_intent_time if use_knife else punch_intent_time
			await get_tree().create_timer(delay).timeout
			
			#top_state_switch_to(top_states.ATTACK_MELEE)
			if is_flinching:
				is_melee_attacking = false
				top_state_switch_to(top_states.ALERT)
			elif not is_flinching:
				top_state_switch_to(top_states.ATTACK_MELEE)
			return
		
		top_states.ATTACK_SHOOT:
			return
		
		top_states.ATTACK_MELEE:
			if not is_attack_ready:
				return
			
			if not is_melee_attacking or not active_target:
				gradual_velo_stop()
				top_state_switch_to(top_states.ALERT)
				return
			
			sound_melee_swipe.play()
			
			var hit_successful: bool = false
			var target_distance: float = get_target_distance()
			if target_distance < melee_distance_threshold and A.is_actor_ahead_approx(self, active_target) and target_distance > 0:
				if "is_crouching" in active_target and active_target.is_crouching:
					pass
				else:
					active_target.hit(melee_damage, U.hit_types.UNDEFINED, self, rhand_anchor.global_position)
					hit_successful = true
				if use_knife:
					A.create_ghost_impact(self, active_target, 0)
				
				var thing_to_say: String = U.random_choice(
					[
						"fun",
						"kill",
						"woo",
						"please"
					]
				)
				speak(thing_to_say, 0.7)
			
			else:
				var melee_collider: Node3D = strike_raycast.get_collider()
				if (
					melee_collider and \
					melee_collider.is_in_group("hit_takers") and \
					is_in_melee_range(melee_collider.global_position)
				):
					melee_collider.hit(melee_damage, U.hit_types.UNDEFINED, self, rhand_anchor.global_position)
					hit_successful = true
					await get_tree().create_timer(0.1).timeout # Give enough time for the animation to play?
			
			if hit_successful:
				sound_melee_hit.play()
			is_melee_attacking = false
			is_recently_attacking = true
			is_attack_ready = false
			
			var t_recent_attack_expire = get_tree().create_tween()
			t_recent_attack_expire.tween_callback(func(): is_recently_attacking = false).set_delay(extended_attack_cooloff)
			
			var t_attack_ready_expire = get_tree().create_tween()
			t_attack_ready_expire.tween_callback(func(): is_attack_ready = true).set_delay(melee_cooloff)
			return
		
		top_states.DEAD:
			return
		
	return

func is_in_melee_range(target_position: Vector3):
	return self.global_position.distance_to(target_position) < melee_distance_threshold

func get_target_distance():
	if not active_target:
		return -1
	return self.global_position.distance_to(active_target.global_position)

func update_active_target():
	# Get the closest target and set it as the active target
	target_board.clear()
	
	var target_candidates: Array = []
	for group in groups_i_dont_like:
		var group_nodes: Array = get_tree().get_nodes_in_group(group)
		if not group_nodes:
			continue
		target_candidates.append_array(group_nodes)
	
	if not target_candidates:
		return
	
	for candidate in target_candidates:
		var distance: float = self.global_position.distance_to(candidate.global_position)
		if distance > target_distance_threshold:
			continue
		target_board[candidate] = {
			"distance": distance,
			"name": candidate.name,
			"instance_id": candidate.get_instance_id()
		}
	
	if not target_board:
		passive_target = active_target
		active_target = null
		return
	
	var closest_target: Dictionary = {"distance": -1}
	for target in target_board.keys():
		if closest_target["distance"] < 1:
			closest_target = target_board[target]
			continue
		elif closest_target["distance"] < target_board[target]["distance"]:
			closest_target = target_board[target]
			continue
	
	if is_instance_of(instance_from_id(closest_target["instance_id"]), CharacterBody3D):
		active_target = instance_from_id(closest_target["instance_id"])
	else:
		active_target = null
		return
	if not random_remembered_target_position:
		random_remembered_target_position = active_target.global_position
	else:
		var remember_possibility: float = .5
		if randf_range(0.0, 0.99) < remember_possibility:
			random_remembered_target_position = active_target.global_position
	return

func move_nav(face_direction: bool = true, new_position: Vector3 = Vector3.ZERO):
	#if not nav_agent.is_target_reachable():
		#return
	if is_being_thrown:
		return
	
	if new_position:
		nav_agent.target_position = new_position
	var next_position: Vector3 = nav_agent.get_next_path_position()
	if face_direction:
		A.face_position(self, next_position)
		A.apply_move(self, Vector2(0, -1), move_speed)
	elif not face_direction:
		self.velocity = (next_position - self.global_position).normalized() * move_speed
	return

func move_feeler_control(lateral_noise: bool = false):
	var left_collider: Node3D = left_feeler.get_collider()
	var right_collider: Node3D = right_feeler.get_collider()
	
	if not left_collider and not right_collider and not is_fastboy:
		move_speed = BASE_MOVE_SPEED # Dangerous setting this here!
		return
	
	if left_collider and left_collider.is_in_group("actors") and not is_fastboy:
		var min_speed: float = move_speed * .8
		var max_speed: float = move_speed * 1.2
		move_speed = randf_range(min_speed, max_speed)
	
	if lateral_noise:
		var lateral_scale: float = 5.0
		var direction = (self.transform.basis * Vector3(randf_range(-lateral_scale, lateral_scale), 0.0, 0.0)).normalized()
		self.velocity.x += direction.x * move_speed
		self.velocity.z += direction.z * move_speed
		
		#var direction = (self.transform.basis * Vector3(randf_range(-lateral_scale, lateral_scale), 0.0, 0.0))
		#self.velocity.x += direction.x * move_speed
		#self.velocity.z += direction.z * move_speed
		
	return

func gradual_velo_stop(stop_vec: Vector3 = Vector3.ZERO, duration: float = 0.25):
	var tween: Tween = get_tree().create_tween()
	tween.tween_property(self, "velocity", stop_vec, duration)
	return

func random_turn(max_angle: float = 45.0):
	self.rotate_y(deg_to_rad(randf_range(max_angle * -1, max_angle)))
	return

func speak(contains_string: String, probability: float = 1.0, force: bool = false):
	if is_recently_speaking and not force:
		return
	
	var probability_triggered: bool = randf_range(0.00, 0.99) < probability
	if probability != 1.0 and not probability_triggered:
		return
	
	var blacklisted_sounds: Array[String] = []
	for sound_name in Blackboard.dump_table["recent_punk_sounds"]:
		if sound_name in blacklisted_sounds:
			continue
		blacklisted_sounds.append(sound_name)
	
	var sound_pool = crazy_sounds_container if is_crazy else normal_sounds_container
	var sound_candidates: Array = []
	for sound_node in sound_pool.find_children("*" + contains_string + "*"):
		if sound_node.name not in blacklisted_sounds:
			sound_candidates.append(sound_node)
	
	if not sound_candidates:
		return
	
	var sound_choice: AudioStreamPlayer3D = U.random_choice(sound_candidates)
	sound_choice.play()
	speech_used.emit(sound_choice.name)
	Blackboard.dump_table["recent_punk_sounds"].push_front(sound_choice.name)
	
	is_recently_speaking = true
	await get_tree().create_timer(speak_cooloff).timeout
	is_recently_speaking = false
	
	await get_tree().create_timer(6.0).timeout
	if sound_choice.name in Blackboard.dump_table["recent_punk_sounds"]:
		Blackboard.dump_table["recent_punk_sounds"].erase(sound_choice.name)
	return sound_choice.name

func aggro_nearby_punks():
	if not active_target:
		push_warning("Tried to aggro punks but no active target.")
		return
	for world_punk in get_tree().get_nodes_in_group("punks"):
		if world_punk == self:
			continue
		var buddy_distance: float = self.global_position.distance_to(world_punk.global_position)
		var aggro_proximity: float = 45.0
		if buddy_distance > aggro_proximity:
			continue
		
		world_punk.active_target = self.active_target
		world_punk.top_state_switch_to(world_punk.top_states.ALERT)
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	if is_dead: return # If no dead flinch, uncomment
	
	var closest_hitbox: CollisionShape3D
	
	health -= damage
	is_flinching = true
	if health > 0:
		U.random_choice(
			$Sounds/voice/Extra.find_children("ouch_*", "AudioStreamPlayer3D", false)
		).play()
	
	if caller and not is_dead:
		var caller_distance: float = self.global_position.distance_to(caller.global_position)
		if caller_distance > target_distance_threshold:
			target_distance_threshold = caller_distance
		active_target = caller
		if not is_being_thrown:
			A.face_position(self, active_target.global_position)
			is_recently_turned = true
			await get_tree().create_timer(look_around_cooloff).timeout
			is_recently_turned = false
		top_state_switch_to(top_states.ALERT)
	
	if health < 1:
		kill_actor()
		
		var caller_groups: Array
		if caller:
			caller_groups = caller.get_groups()
		var world_punks: Array = get_tree().get_nodes_in_group("punks")
		for group in caller_groups:
			if group not in self.get_groups() and not group.contains("_"):
				for world_punk in world_punks:
					if group in world_punk.groups_i_dont_like:
						continue
					world_punk.groups_i_dont_like.append(group)
	
	await get_tree().create_timer(0.5).timeout
	is_flinching = false
	return

func kill_actor():
	if is_dead:
		return
	gradual_velo_stop()
	is_dead = true
	U.random_choice(die_sounds).play()
	
	#for group in ["players", "player_minions"]:
	for group in ["actors"]: # lazy fix for anim twitch tripping over bodies
		for p in get_tree().get_nodes_in_group(group):
			self.add_collision_exception_with(p)
	
	top_state_switch_to(top_states.DEAD)
	return

func on_behavior_process_tick():
	if is_dead and top_state == top_states.DEAD:
		return
	
	behavior_control()
	
	return

func _speech_used(speech_name: String):
	var aggro_texts: Array[String] = [
		"scab",
		"get_em",
		"get_this"
	]
	for aggro_text in aggro_texts:
		if speech_name.contains(aggro_text):
			aggro_nearby_punks()
	return


func _on_alert_fatigue_timeout():
	if top_state == top_states.ALERT:
		top_state_switch_to(top_states.IDLE)
	return

func mod_interact(caller: Node3D):
	prints(self.name, "mod_interact from", caller.name)
	if caller.is_carrying_actor:
		return
	if is_flinching or top_state != top_states.ATTACK_MELEE_INTENT:
		pre_carried_rotation = self.global_rotation
		caller.carried_actor = self
		is_being_carried = true
		#else:
			# add the animation somehow. There's a .res file
			# I got lazy and just added it to the animationplayer manually
		top_state_switch_to(top_states.NONE)
		#var carried_anim: String = "lifter_being_carried"
		#if anim.has_animation(carried_anim):
			#anim.play(carried_anim)
	return

func stop_being_carried(caller: Node3D):
	#self.global_rotation = pre_carried_rotation
	if not is_dead:
		top_state_switch_to(top_states.IDLE)
	else:
		top_state_switch_to(top_states.DEAD)
	is_being_carried = false
	
	is_being_thrown = true
	#get_tree().create_tween().tween_callback(func(): sound_throw_land.play()).set_delay(0.6)
	await get_tree().create_timer(2.2).timeout
	is_being_thrown = false
	
	
	if is_dead:
		A.animate(self, "die_1")
	
	self.global_rotation.x = 0.0
	self.global_rotation.z = 0.0
	
	return

func start_dancing():
	top_state_switch_to(top_states.NONE)
	A.animate(self, "gesture_punk_dance_1", true)
	return

func stop_dancing():
	top_state_switch_to(top_states.IDLE)
	return
