extends CharacterBody2D

signal died(caller)

var health: int = 25
@onready var sprite: Sprite2D = $Sprite2D
@onready var anim: AnimationPlayer = $AnimationPlayer
@onready var raycast_container: Node2D = $raycast_container
@onready var punch_raycast: RayCast2D = $raycast_container/punch_raycast
@onready var kick_raycast: RayCast2D = $raycast_container/kick_raycast
var punch_damage: int = 5
var kick_damage: int = 8
const SPEED = 150.0 # 300.0
const JUMP_VELOCITY = -400.0

# Get the gravity from the project settings to be synced with RigidBody nodes.
var gravity = ProjectSettings.get_setting("physics/2d/default_gravity")
enum {
	DIR_LEFT,
	DIR_RIGHT
}
var facing_direction: int = DIR_RIGHT
var is_punching: bool = false
var is_kicking: bool = false
var is_flinching: bool = false
var is_moving: bool = false
var is_attacking: bool = false
var is_dead: bool = false
var reaction_time: float = 0.8

enum b_states {
	IDLE,
	ALERT,
	ATTACKING,
	DEAD,
	NONE
}
var b_state: int = b_states.IDLE

var active_target: Node2D
var sound_controller: Node

var time_in_air: float = 0.0


func _ready():
	for raycast in raycast_container.get_children():
		raycast.add_exception(self)
	for thug in get_tree().get_nodes_in_group("arcade_thugs"):
		self.add_collision_exception_with(thug)
	return

func _physics_process(delta):
	
	$Label.text = str(health)
	if is_dead:
		return
	
	if not is_on_floor():
		velocity.y += gravity * delta
		time_in_air += delta
	
	behavior_control()
	animation_control()
	move_and_slide()
	return

func behavior_control():
	match b_state:
		b_states.IDLE:
			update_active_target()
			if not active_target:
				return
			
			if active_target:
				switch_to_state(b_states.ALERT)
			return
		b_states.ALERT:
			#if active target and self.global_position.y > active_target.global_position.y and not active_target.is_dead:
			if active_target and (self.global_position.y > active_target.global_position.y) and not active_target.is_dead:
				apply_move(get_target_direction() * -1)
				return
			
			if U.coin_flip():
				is_moving = true
				get_tree().create_tween().tween_callback(
					func(): is_moving = false
				).set_delay(.8)
			
			if is_moving:
				var move_dir: float = get_target_direction()
				apply_move(
					move_dir
				)
				if move_dir < 0:
					set_facing_dir(DIR_LEFT)
				elif move_dir > 0:
					set_facing_dir(DIR_RIGHT)
			
			if active_target and active_target.global_position.distance_to(self.global_position) < 100:
				switch_to_state(b_states.ATTACKING)
			return
		b_states.ATTACKING:
			if is_punching or is_kicking or is_flinching or is_attacking:
				return
			
			is_attacking = true
			await get_tree().create_timer(reaction_time).timeout
			punch()
			await get_tree().create_timer(1.0).timeout
			is_attacking = false
			switch_to_state(b_states.ALERT)
			return
		b_states.DEAD:
			return
		b_states.NONE:
			return
	return

func hit(caller: Node2D, damage: int):
	health -= damage
	if health < 1:
		kill_actor()
	if health > 0:
		flinch()
	return

func punch():
	if is_punching or is_kicking or is_flinching: return
	
	is_punching = true
	get_tree().create_tween().tween_callback(
		func(): if not is_flinching: is_punching = false
	).set_delay(.2)
	if punch_raycast.get_collider():
		var collider: Object = punch_raycast.get_collider()
		collider.hit(self, punch_damage)
	return

func kick():
	is_kicking = true
	get_tree().create_tween().tween_callback(
		func(): if not is_flinching: is_kicking = false
	).set_delay(.2)
	if punch_raycast.get_collider():
		var collider: Object = punch_raycast.get_collider()
		collider.hit(self, kick_damage)
	return

func flinch():
	is_flinching = true
	get_tree().create_tween().tween_callback(
		func(): is_flinching = false
	).set_delay(.2)
	return

func kill_actor():
	
	if not sound_controller: set_sound_controller()
	sound_controller.playsound("thug_die")
	
	is_dead = true
	$CollisionShape2D.disabled = true
	switch_to_state(b_states.DEAD)
	animation_control()
	
	died.emit(self)
	return

func get_target_direction() -> float:
	if not active_target:
		update_active_target()
	if not active_target: # Still
		return 0
	
	return (active_target.position - self.position).normalized().x

func update_active_target():
	var player: Node2D = get_tree().get_first_node_in_group("arcade_players")
	if player:
		active_target = player
	return

func switch_to_state(new_state: int):
	b_state = new_state
	return

func apply_move(direction: float):
	## Handle jump.
	#if Input.is_action_just_pressed("ui_accept") and is_on_floor():
		#velocity.y = JUMP_VELOCITY
#
	## Get the input direction and handle the movement/deceleration.
	## As good practice, you should replace UI actions with custom gameplay actions.
	#var direction = Input.get_axis("ui_left", "ui_right")
	if direction:
		velocity.x = direction * SPEED
	else:
		velocity.x = move_toward(velocity.x, 0, SPEED)
	return

func animation_control():
	if is_dead:
		self.velocity = Vector2.ZERO
		anim.play("die")
		return
	
	if is_flinching:
		anim.play("flinch")
		return
	
	if is_kicking:
		anim.play("kick")
		return
	
	if is_punching:
		anim.play("punch")
		return
	
	if self.velocity.length() > 0:
		anim.play("walk")
	else:
		anim.play("stand")
	return

func set_facing_dir(new_dir: int):
	facing_direction = new_dir
	if facing_direction == DIR_LEFT:
		if sprite.scale.x > 0:
			sprite.scale.x *= -1
			raycast_container.scale.x *= -1
	if facing_direction == DIR_RIGHT:
		if sprite.scale.x < 0:
			sprite.scale.x *= -1
			raycast_container.scale.x *= -1
	return


func set_sound_controller():
	sound_controller = get_tree().get_first_node_in_group("CRIMEBUSTER_sound_controller")
	return
