extends Control

@onready var prompt: LineEdit = $bar_edit
@onready var output_line_container: VBoxContainer = $vbox_output_lines
@onready var output_line_template: Label = $vbox_output_lines/prompt_space
#var output_lines: Array = []
var output_die_time: float = 0.00
var output_line_limit: int = 16
enum command_types {
	SINGLE,
	ARGS
}
@onready var command_bar_elements: Array = [
	output_line_container,
	prompt,
	$bar_bg,
	#$ColorRect
]
var is_enabled: bool = true
var cmd_history: Array = []
var cmd_history_limit: int = 24
var cmd_history_index: int = 0

func _ready():
	if not U.is_node_root(self.get_parent()):
		disable()
	else:
		enable()
	
	return

func _input(event):
	if not is_enabled:
		if event.is_action_pressed("dev_button"):
			self.enable()
		return
	
	if event.is_action_pressed("ui_cancel"):
		self.disable()
	
	if event.is_action_pressed("ui_up"):
		cmd_history_up()
	
	
	return

func _on_bar_edit_text_submitted(new_text):
	if output_line_container.get_child_count() > output_line_limit:
		output_line_container.get_children()[0].free()
	var new_output_line: Label = output_line_template.duplicate()
	output_line_container.add_child(new_output_line)
	output_line_template.move_to_front()
	
	new_output_line.text = interpreter(new_text)
	
	prompt.clear()
	if output_die_time > 0.00:
		await get_tree().create_timer(output_die_time).timeout
		new_output_line.queue_free()
	return

func interpreter(new_command: String):
	var output: String
	if new_command.contains(" "):
		output = interpret_arg_command(new_command)
	else:
		output = interpret_single_command(new_command)
	
	cmd_history.append(new_command)
	if len(cmd_history) > cmd_history_limit:
		cmd_history.pop_back()
	return output

func interpret_single_command(new_command: String):
	var output: String
	match new_command:
		"kill":
			player_suicide()
			output = "Self terminated"
		"suicide":
			player_suicide()
			output = "Self terminated"
		"quit":
			output = "Killing session"
			get_tree().quit()
		"exit":
			output = "Killing session"
			get_tree().quit()
		"melt":
			$ColorRect.generate_offsets()
			#get_tree().create_tween().tween_callback(func(): $ColorRect.transition()).set_delay(1.0)
			$ColorRect.transition()
		"reload":
			get_tree().get_first_node_in_group("game_tree").restart_level()
			self.disable()
			get_tree().create_tween().tween_callback(func():
				self.enable()
			).set_delay(.9)
		"coinflip":
			output = "Heads" if U.coin_flip() else "Tails"
		"endscene":
			Blackboard.current_world.scene_ended.emit()
	return output

func interpret_arg_command(new_command: String):
	var output: String
	var raw_split: PackedStringArray = new_command.split(" ")
	var command: String = raw_split[0]
	var tokens: Array[String] = []
	var arg_chunk: String = new_command.replace(command + " ", "")
	for piece in raw_split.slice(1):
		tokens.append(piece)
	
	match command:
		"echo":
			output = arg_chunk
		"move":
			output = cmd_move_node(arg_chunk)
		"map":
			output = cmd_load_map(arg_chunk)
		"spawn":
			output = cmd_spawn(arg_chunk)
		"volume":
			output = cmd_volume(arg_chunk)
		"timescale":
			output = cmd_timescale(arg_chunk)
		"ghost":
			output = cmd_ghost(arg_chunk)
		"showfps":
			output = cmd_showfps(arg_chunk)
		#"load":
			#pass
	
	return output

func disable():
	var player: Node3D = get_tree().get_first_node_in_group("players")
	if player:
		player.is_input_allowed = true
	for element in command_bar_elements:
		element.visible = false
	self.visible = false
	await get_tree().create_timer(.1).timeout
	is_enabled = false
	return

func enable():
	var player: Node3D = get_tree().get_first_node_in_group("players")
	if player:
		player.is_input_allowed = false
	
	for element in command_bar_elements:
		element.visible = true
	self.visible = true
	is_enabled = true
	await get_tree().process_frame
	prompt.grab_focus()
	return

func cmd_history_up():
	prompt.text = cmd_history[cmd_history_index]
	cmd_history_index += 1
	if cmd_history_index > (len(cmd_history) - 1):
		cmd_history_index = 0
	return

func player_suicide():
	var player: Node3D = get_tree().get_first_node_in_group("players")
	player.hit(9999, U.hit_types.UNDEFINED, player, player.global_position)
	return

func cmd_move_node(arg_content: String):
	# "move @player 203 6 23.2"
	# "move @player +0 +35 +0"
	var command_output: String = "cmd_move_node: undefined"
	var tokens: Array = arg_content.split(" ")
	prints(self.name, "cmd_move_node:", tokens)
	var tok_target_node: String = tokens[0]
	var tok_move_x: String = tokens[1]
	var tok_move_y: String = tokens[2]
	var tok_move_z: String = tokens[3]
	
	var target_node: Node3D
	if tok_target_node.begins_with("@"):
		target_node = convert_object_alias(tok_target_node)
	else:
		target_node = get_tree().find_children(tok_target_node, "Node3D", true, false)
	
	var destination_vector: Vector3 = Vector3.ZERO
	destination_vector.x = float(tok_move_x) if not tok_move_x.begins_with("+") else target_node.global_position.x + float(tok_move_x.replace("+", ""))
	destination_vector.y = float(tok_move_y) if not tok_move_y.begins_with("+") else target_node.global_position.y + float(tok_move_y.replace("+", ""))
	destination_vector.z = float(tok_move_z) if not tok_move_z.begins_with("+") else target_node.global_position.z + float(tok_move_z.replace("+", ""))
	
	target_node.global_position = destination_vector
	command_output = "Moved " + str(target_node) + " to " + str(destination_vector)
	return command_output

func convert_object_alias(alias_name: String):
	var raw_alias: String = alias_name.replace("@", "")
	match raw_alias:
		"player":
			return get_tree().get_first_node_in_group("players")
	return null

func cmd_load_map(arg_content: String):
	var command_output: String = "cmd_load_map: undefined"
	var is_successful: bool = false
	var tokens: Array = arg_content.split(" ")
	var scene_query: String = tokens[0]
	var scene_path: String = ""
	var scene_prefixes: Array[String] = [
		"res://Scenes/Levels/",
		"res://Scenes/",
	]
	var scene_suffix: String = ".tscn"
	for scene_prefix in scene_prefixes:
		var guessed_path: String = scene_prefix + scene_query
		if not guessed_path.contains(scene_suffix):
			guessed_path = guessed_path + scene_suffix
		
		if ResourceLoader.exists(guessed_path):
			scene_path = guessed_path
			break
	
	if not scene_path:
		command_output = "Scene path '" + scene_query + "' not found."
		return command_output
	
	if U.grptop("game_tree"):
		U.grptop("game_tree").set_active_scene(scene_path)
		command_output = "Loading '" + scene_path + "'"
		is_successful = true
	elif not U.grptop("game_tree") and U.grptop("main_menu"):
		Director.startup_type = Director.startup_types.LOADED
		Director.current_scene_path = scene_path
		U.grptop("main_menu").start_loaded_game()
		command_output = "Loading '" + scene_path + "'"
		is_successful = true
	else:
		command_output = "No base 'main_menu' or 'game_tree' to load from."
	if is_successful:
		Director.add_load_run(self, "disable")
	return command_output

func cmd_spawn(arg_content: String):
	var command_output: String = "cmd_spawn: undefined"
	# Spawnables:
	# - Actors (wiseguy.tscn, punk.tscn, esc.)
	# - Weapons (w_*.tscn)
	# - Items (health_flask.tscn)
	return command_output

func cmd_volume(arg_content: String):
	var command_output: String = "cmd_volume: undefined"
	var is_successful: bool = false
	var tokens: Array = arg_content.split(" ")
	var new_value: float
	if tokens[0] == "default":
		new_value = Global.default_master_volume
	else:
		new_value = float(tokens[0])
	AudioServer.set_bus_volume_db(0, new_value)
	command_output = "Volume set to: " + str(new_value)
	return command_output

func cmd_timescale(arg_content: String):
	var command_output: String = "cmd_timescale: undefined"
	var is_successful: bool = false
	var tokens: Array = arg_content.split(" ")
	var new_value: float
	if tokens[0] == "default" or tokens[0] == "reset":
		new_value = 1.0
	else:
		new_value = float(tokens[0])
	Engine.time_scale = new_value
	command_output = "Timescale set to: " + str(new_value)
	return command_output

func cmd_ghost(arg_content: String):
	var command_output: String = "cmd_ghost: undefined"
	var is_successful: bool = false
	var tokens: Array = arg_content.split(" ")
	var new_value: String
	new_value = tokens[0]
	if new_value.to_lower() == "on":
		get_tree().get_first_node_in_group("players").set_as_debug_ghost(true)
		get_tree().get_first_node_in_group("dev").is_input_allowed = true
		
	elif new_value.to_lower() == "off":
		get_tree().get_first_node_in_group("dev").set_as_debug_ghost(false)
	command_output = "Ghost set to: " + str(new_value)
	return command_output

func cmd_showfps(arg_content: String):
	var command_output: String = "cmd_showfps: undefined"
	var is_successful: bool = false
	var tokens: Array = arg_content.split(" ")
	var new_value: String
	new_value = tokens[0]
	if new_value.to_lower() == "on":
		get_tree().get_first_node_in_group("debug_fps_counter").visible = true
		
	elif new_value.to_lower() == "off":
		get_tree().get_first_node_in_group("debug_fps_counter").visible = false
	command_output = "showfps set to: " + str(new_value)
	return command_output
