extends CharacterBody3D

@export var custom_audio: AudioStream
@onready var custom_audio_player: AudioStreamPlayer3D = $Sounds/custom_audio_player
@onready var anim: AnimationPlayer = $merc_vela/AnimationPlayer
@onready var eyes: Node3D = $mid_anchor/eyes
var ticker_interval: float = .05
var accumulated_ticker: float = 0.0
var assumed_player: PhysicsBody3D
var is_speaking: bool = false
var speak_ready: bool = false
var is_interrupted: bool = false

enum speak_states {
	SILENT,
	SCRIPTED,
	INTERRUPT
}
var speak_state: speak_states = speak_states.SILENT
var previous_speak_state: speak_states = speak_state

var audio_seek_place: float = 0.0
var queued_speak_node: AudioStreamPlayer3D
var active_speak_node: AudioStreamPlayer3D
var speak_sequence_name: String = ""
var speak_sequence_index: int = 0

# Called when the node enters the scene tree for the first time.
func _ready() -> void:
	if custom_audio:
		custom_audio_player.stream = custom_audio
	return


# Called every frame. 'delta' is the elapsed time since the previous frame.
func _process(delta: float) -> void:
	accumulated_ticker += delta
	if accumulated_ticker >= ticker_interval:
		tick_control()
		speak_control()
		accumulated_ticker = 0.0
	return

func animation_control() -> void:
	if is_speaking:
		A.animate(self, "cinem_gesture_talk_1")
		return
	
	A.animate(self, "cinem_idle_1")
	return

func tick_control() -> void:
	if not assumed_player:
		assumed_player = get_tree().get_first_node_in_group("players") 
	if A.distance(self, assumed_player) < 19.0:
		A.face_position(self, assumed_player.global_position)
	animation_control()
	return

func speak_control() -> void:
	match speak_state:
		speak_states.SILENT:
			if speak_ready:
				speak_state_switch_to(speak_states.SCRIPTED)
				return
			
			# Maybe a bad idea?
			if not is_speaking and is_interrupted:
				speak_state_switch_to(speak_states.INTERRUPT)
			return
		speak_states.SCRIPTED:
			if not active_speak_node:
				prints(self.name, "SCRIPTED state with no speak node")
				return
			if is_speaking:
				if is_interrupted:
					audio_seek_place = active_speak_node.get_playback_position()
					active_speak_node.stop()
					is_speaking = false
					speak_state_switch_to(speak_states.INTERRUPT)
				return
			is_speaking = true
			if not active_speak_node.playing:
				stop_all_audio()
				active_speak_node.play(audio_seek_place)
			await get_tree().create_timer(active_speak_node.stream.get_length()).timeout
			
			if speak_state == speak_states.INTERRUPT or is_interrupted:
#				speak_state_switch_to(speak_states.SCRIPTED)
				return
			
			is_speaking = false
			speak_ready = false
			speak_state_switch_to(speak_states.SILENT)
			Director.speak_sequence_ended(speak_sequence_name, speak_sequence_index)
			return
		speak_states.INTERRUPT:
			if is_speaking:
				return
			is_speaking = true
			var interrupt_sound: AudioStreamPlayer3D = $Sounds.find_children("interrupt_*").pick_random()
			interrupt_sound.play()
			await get_tree().create_timer(interrupt_sound.stream.get_length()).timeout
			await get_tree().create_timer(.1).timeout
			if queued_speak_node and not active_speak_node:
				active_speak_node = queued_speak_node
				queued_speak_node = null
			#speak_state_switch_to(speak_states.SCRIPTED)
			speak_state_switch_to(previous_speak_state)
			is_speaking = false
			is_interrupted = false
			return
	return

func speak_state_switch_to(new_state: speak_states) -> void:
	previous_speak_state = speak_state
	speak_state = new_state
	return

func sequence_speak_trigger(sequence_name: String = "", sequence_index: int = 0) -> void:
	speak_sequence_name = sequence_name
	speak_sequence_index = sequence_index
	var speak_node_query: String = Director.speaking_sequences[sequence_name][sequence_index]["node_query"]
	audio_seek_place = 0.0
	if speak_state == speak_states.INTERRUPT:
		queued_speak_node = $Sounds.find_child(speak_node_query)
		active_speak_node = null
	else:
		active_speak_node = $Sounds.find_child(speak_node_query)
		speak_ready = true
	return

func interrupt_bad_touch(thing: Node3D) -> void:
	if is_interrupted:
		return
	is_interrupted = true
	return

func stop_all_audio() -> void:
	$Sounds.find_children("*", "AudioStreamPlayer3D").map(func (n): n.stop())
	return

func activate_custom_audio() -> void:
	stop_all_audio()
	is_speaking = true
	custom_audio_player.play()
	await custom_audio_player.finished
	is_speaking = false
	return
