extends CharacterBody3D

# DIFFICULTIES
# Normal
# - attack_warn_timer = 1.0
# - roam_timer = 3.0
# "Ultra Violence"
# - attack_warn_timer = 0.5
# - roam_timer = 1.0

## LEFT OFF
# - Animate/react differently to hard hits (like the Fifty or a shotty)
# - Check for nearby items when hit and dying. Fall out of windows, break tables, fall off rails, etc.
# - Make them more interesting when they're in groups

@onready var debug_label: Label3D = $debug_label
@onready var anim: AnimationPlayer = $model/AnimationPlayer
@onready var nav_agent: NavigationAgent3D = $NavigationAgent3D
@onready var eyes: Node3D = $mid_anchor/eyes
@onready var sight_raycast: RayCast3D = $sight_raycast
@onready var strike_raycast: RayCast3D = $strike_raycast
#@onready var rhand_anchor: Node3D = $model/Armature/Skeleton3D/rhand_attachment/anchor
@onready var rhand_anchor: Node3D = $Bones/rhand_bone/anchor
@onready var equip_delay: Timer = $equip_delay
@onready var attack_warn_timer: Timer = $attack_warn_timer
@onready var melee_warn_timer: Timer = $melee_warn_timer
@onready var attack_cooloff: Timer = $attack_cooloff
@onready var roam_timer: Timer = $roam_timer
@onready var flinch_timer: Timer = $flinch_timer
@onready var alert_fatigue: Timer = $alert_fatigue

@onready var hitboxes: Array[CollisionShape3D] = [
	$hitbox_head,
	$hitbox_torso,
	$hitbox_lower,
	$hitbox_knees,
	$hitbox_foot
]

@onready var hitbox_head: CollisionShape3D = $hitbox_head
@onready var hitbox_torso: CollisionShape3D = $hitbox_torso
@onready var hitbox_lower: CollisionShape3D = $hitbox_lower
@onready var hitbox_knees: CollisionShape3D = $hitbox_knees
@onready var hitbox_foot: CollisionShape3D = $hitbox_foot
@onready var ghost_position: Node3D = $model/Armature/Skeleton3D/torso_attachment/anchor/ghost_position

const BASE_MOVE_SPEED = 5.0
var move_speed: float = BASE_MOVE_SPEED
var run_scale = 1.2

const JUMP_VELOCITY = 4.5

@export var health: int = 55

@export var world_queue_on_spawn: bool = true
var is_in_world_queue: bool = false
@export var world_queue_strict: bool = false

@export var spawn_weapon: String
@export var random_outfit: bool = false
@export_range(-1, 100) var roam_chance: float = 100.00
@export var enable_static_roam_chance: bool = true
@export var melee_damage: float = 16.0
@export var melee_recoil: float = -30.0
var static_roam_chance: bool = false
var is_dead: bool = false
var last_anim: String = ""
var queued_anims: Array[String] = []

var active_weapon_name: String
var active_weapon_node: Node3D
var active_weapon_ammo_left: int
var active_weapon_mag_size: int
var is_equipping: bool = false
var is_aiming: bool = false
var is_attacking: bool = false
var is_melee_attacking: bool = false
var is_roaming: bool = false
var is_flinching: bool = false

var ouch_sounds: Array[AudioStreamPlayer3D] = []
var die_sounds: Array[AudioStreamPlayer3D] = []
@export var voice: String = ""

#############################
var behavior_process_ticker: float = 0.0
const FAST_PROCESS_TICK: float = 0.049
const DEFAULT_PROCESS_TICK: float = 0.10
@export var behavior_process_interval: float = FAST_PROCESS_TICK

enum top_states {
	NONE,
	IDLE,
	ALERT,
	MOVING,
	GO_TO,
	ATTACK_INTENT,
	ATTACK,
	MELEE_ATTACK,
	DEAD
}
var top_state: int = top_states.IDLE
var previous_top_state: int = top_states.IDLE

const BASE_TARGET_DISTANCE_THRESHOLD: float = 45.0
var target_distance_threshold: float = BASE_TARGET_DISTANCE_THRESHOLD
var melee_distance_threshold: float = 7.0
#var passive_target: CharacterBody3D
#var active_target: CharacterBody3D
var passive_target: PhysicsBody3D
var active_target: PhysicsBody3D
var last_seen_target_position: Vector3
var strike_position: Vector3
var roam_position: Vector3
var sound_position: Vector3
var target_pool: Array[CharacterBody3D] = []
@export var groups_i_dont_like: Array[String] = [
	"players",
	"player_minions",
	"punks"
]

var sounds_heard: Array = []
@export var enable_sound_following: bool = false
var is_following_sound: bool = false

var go_to_position: Vector3 = Vector3.ZERO
var last_go_to_position: Vector3 = Vector3.ZERO

var double_attack_enabled: bool = false
var double_attack_counter: int = 0
var double_attack_threshold: int = 2

#############################

func _ready():
	nav_agent.path_desired_distance = 5.0
	
	if not spawn_weapon:
		spawn_weapon = "pistol_9mm"
	
	if random_outfit:
		var casual_outfit_meshes: Array = [
			$model/Armature/Skeleton3D/wg_body,
			$model/Armature/Skeleton3D/wg_casual_clothes_1,
			$model/Armature/Skeleton3D/wg_glasses_2,
			$model/Armature/Skeleton3D/wg_hands,
			$model/Armature/Skeleton3D/wg_head
		]
		var regular_outfit_base_meshes: Array = [
			$model/Armature/Skeleton3D/wg_glasses_1,
			$model/Armature/Skeleton3D/wg_hands,
			$model/Armature/Skeleton3D/wg_head
		]
		var hair_meshes: Array = [
			$model/Armature/Skeleton3D/wg_hair_1,
			$model/Armature/Skeleton3D/wg_hair_2,
			$model/Armature/Skeleton3D/wg_hair_3
		]
		for m in $model/Armature/Skeleton3D.get_children():
			m.visible = false
		if U.coin_flip() and U.coin_flip():
			for m in casual_outfit_meshes:
				m.visible = true
		else:
			for m in regular_outfit_base_meshes:
				m.visible = true
			[$model/Armature/Skeleton3D/fed_suit_1, $model/Armature/Skeleton3D/wg_suit_1].pick_random().visible = true
			$model/Armature/Skeleton3D/wg_head_accessories_1.visible = (U.coin_flip() and U.coin_flip())
			hair_meshes.pick_random().visible = true
	else:
		pass
	
	if voice == "random" or voice == "":
		voice = ["voice1", "voice2"][randi_range(0, 1)]
	
	match voice:
		"voice1":
			ouch_sounds = [
				$Sounds/voice1/wiseguy_ouch1,
				$Sounds/voice1/wiseguy_ouch2,
				$Sounds/voice1/wiseguy_ouch3,
				$Sounds/voice1/wiseguy_ouch4,
				$Sounds/voice1/wiseguy_ouch5
			]
			die_sounds = []
		"voice2":
			ouch_sounds = [
				$Sounds/voice2/wiseguy_ouch1,
				$Sounds/voice2/wiseguy_ouch2,
				$Sounds/voice2/wiseguy_ouch3
			]
			die_sounds = [
				$Sounds/voice2/wiseguy_die1,
				$Sounds/voice2/wiseguy_die2
			]
	
	if roam_chance < 0.0:
		roam_chance = randf_range(0.0, 100.00)
		prints(self.name, randf_range(0.0, 100.00))
	if enable_static_roam_chance:
		static_roam_chance = randf_range(0.0, 0.99) < roam_chance
	
# Normal
# - attack_warn_timer = 1.0
# - roam_timer = 3.0
# "Ultra Violence"
# - attack_warn_timer = 0.5
# - roam_timer = 1.0
	match Global.skill_level:
		Global.skill_levels.SWEET_DREAMS:
			attack_warn_timer.wait_time = 1.0
			roam_timer.wait_time = 3.0
		Global.skill_levels.ROUGH_NIGHT:
			attack_warn_timer.wait_time = 0.5
			roam_timer.wait_time = 2.0
		Global.skill_levels.NIGHT_TERROR:
			attack_warn_timer.wait_time = 0.5
			roam_timer.wait_time = 0.75
			move_speed = BASE_MOVE_SPEED * 2.5
			anim.speed_scale = 2.0
		Global.skill_levels.PARALYSIS:
			attack_warn_timer.wait_time = 0.1
			roam_timer.wait_time = 0.5
			#spawn_weapon = "pistol_fifty"
			move_speed = BASE_MOVE_SPEED * 2.5
			double_attack_enabled = true
			var sound_follower_chance: int = randi_range(0, 1)
			if sound_follower_chance:
				enable_sound_following = true
	
	if world_queue_on_spawn:
		Blackboard.add_to_universal_world_queue(self)
	
	return

func _physics_process(delta):
	debug_label.text = self.name + " " + top_states.keys()[top_state]
	if not is_dead:
		tick_control(delta)
	
	if not is_on_floor():
		A.apply_gravity(self, delta)
	
	move_and_slide()
	return

func tick_control(delta):
	behavior_process_ticker += delta
	if behavior_process_ticker > behavior_process_interval:
		if not is_in_world_queue:
			on_behavior_process_tick()
		behavior_process_ticker = 0.0
	return

func behavior_control():
	match top_state:
		top_states.NONE:
			return
			
		top_states.IDLE:
			update_targets_by_distance()
			if not is_flinching:
				anim_h("unarmed_idle")
			
			if not active_target:
				world_queue_checkin()
				return
			if active_target:
				top_state_switch_to(top_states.ALERT)
				if is_in_world_queue and not world_queue_strict:
					Blackboard.remove_from_universal_world_queue(self)
			
			sounds_heard = A.check_for_sounds(self, target_distance_threshold)
			for sound in sounds_heard:
				var sound_parent: Node3D = sound.get_parent()
				if sound_parent in target_pool:
					A.face_position(self, sound_parent.global_position)
					if enable_sound_following:
						sound_position = sound.global_position
						is_following_sound = true
						alert_fatigue.start()
						break
			if not active_weapon_node:
				equip_weapon(spawn_weapon)
				return
				
			if is_target_viewable():
				top_state_switch_to(top_states.ATTACK_INTENT)
				return
			
			if is_following_sound:
				nav_agent.target_position = sound_position
				top_state_switch_to(top_states.ALERT)
				return
			return
			
		top_states.ALERT:
			#anim_h("armed_idle")
			
			update_targets_by_distance()
			if not active_target and not is_following_sound:
				prints(self.name, active_target)
				self.rotate_y(deg_to_rad(25))
				top_state_switch_to(top_states.IDLE)
				return
			
			var target_distance: float
			if active_target and is_following_sound:
				target_distance = self.global_position.distance_to(sound_position)
			elif active_target and not is_following_sound:
				target_distance = self.global_position.distance_to(active_target.global_position)
			elif not active_target:
				top_state_switch_to(top_states.IDLE)
			
			if target_distance < 10:
				roam_timer.stop()
				#roam_timer.emit_signal("timeout")
				top_state_switch_to(top_states.ATTACK_INTENT)
				return
			
			if is_roaming:
				#A.apply_move(self, Vector2.UP, move_speed)
				if active_target and not roam_position:
					roam_position = active_target.global_position
				elif not active_target and not roam_position:
					roam_position = sound_position
				
				#if not roam_position:
					### What was I thinking? Face a non-existing position?
					#var lateral_noise: float = randf_range(-15.0, 15.0)
					#roam_position.x += lateral_noise
					#roam_position.z += lateral_noise
					#A.face_position(self, roam_position)
				
				if health < 20:
					A.apply_move(self, Vector2.DOWN, move_speed)
					nav_agent.target_position = roam_position
				else:
					nav_agent.target_position = roam_position
				var nav_distance: float = self.global_position.distance_to(nav_agent.target_position)
				if nav_distance < 7.0:
					roam_timer.stop()
					roam_timer.emit_signal("timeout")
				else:
					move_nav(true)
					anim_h("armed_run")
				return
			
			if not is_roaming and is_following_sound and self.global_position.distance_to(sound_position) < 5.0:
				top_state_switch_to(top_states.IDLE)
				return
			
			if not is_roaming and is_following_sound:
				var next_position: Vector3 = nav_agent.get_next_path_position()
				var move_direction: Vector3 = (next_position - self.global_position).normalized()
				self.velocity = move_direction * move_speed
				anim_h("armed_run")
				return
			
			if is_roam_chance_triggered():
				is_roaming = true
				roam_timer.start()
				#anim_h("armed_run")
			else:
				top_state_switch_to(top_states.IDLE)
			return
		
		top_states.MOVING:

			return
		
		top_states.GO_TO:
			if not go_to_position:
				prints(self.name, "No go_to_position. Switching to previous state.", top_states.keys()[previous_top_state], "->", top_states.keys()[top_state], "->", top_states.keys()[previous_top_state])
				top_state_switch_to(previous_top_state)
			
			
			var is_destination_reached: bool = self.global_position.distance_to(go_to_position) < 5.0
			if is_destination_reached:
				prints(self.name, "Destination reached.", top_states.keys()[top_state], "->", top_states.keys()[previous_top_state])
				top_state_switch_to(previous_top_state)
				go_to_position = Vector3.ZERO
			else:
				A.face_position(self, nav_agent.get_next_path_position())
				A.apply_move(self, Vector2.UP, move_speed)
			return
		
		top_states.ATTACK_INTENT:
			if is_aiming or is_equipping and not attack_warn_timer.is_stopped():
				return
			
			if not active_weapon_node:
				equip_weapon(spawn_weapon)
				return
			
			# - Is there at least one bullet in it?
			# 	- If not, reload weapon
			if not active_target:
				top_state_switch_to(top_states.IDLE)
				return
			# Telegraph attack to target, then switch to ATTACK
			# 	Point at the last place you saw the target, the attack will happen there.
			if is_target_viewable():
				strike_position = active_target.global_position
			else:
				if not is_flinching:
					await get_tree().create_timer(0.5).timeout
				if active_target and active_target.global_position.distance_to(self.global_position) > 10:
					top_state_switch_to(top_states.ALERT)
				else:
					A.apply_move(self, Vector2.DOWN, move_speed)
				return
			
			if is_target_in_melee_range():
				A.face_position(self, strike_position)
				anim_h("melee_attack_1_warn", false, true)
				is_aiming = true
				
				melee_warn_timer.start()
			else:
				self.velocity = Vector3.ZERO
				if not active_target.is_in_group("players"):
					strike_position.y += 5.0
				A.face_position(self, strike_position)
				strike_raycast.look_at(strike_position)
				anim_h("aim_focus")
				is_aiming = true
				
				attack_warn_timer.start()
			return
		
		top_states.ATTACK:
			if is_attacking:
				return
			if is_flinching:
				return
			
			if double_attack_enabled:
				self.velocity = Vector3.ZERO
				strike_target(active_target)
				double_attack_counter += 1
				await get_tree().create_timer(.25).timeout
				strike_target(active_target)
				await get_tree().create_timer(.25).timeout
				top_state_switch_to(top_states.ALERT)
				return
			
			self.velocity = Vector3.ZERO
			strike_target(active_target)
			return
		
		top_states.MELEE_ATTACK:
			if is_melee_attacking:
				return
			
			if attack_cooloff.is_stopped():
				is_attacking = true
				attack_cooloff.start()
			
			if not is_target_in_melee_range():
				return
			
			is_melee_attacking = true
			anim_h("melee_attack_1", false, true)
			$Sounds/punch_swipe.play()
			if A.is_actor_ahead_approx(self, active_target):
				active_target.hit(melee_damage, U.hit_types.UNDEFINED, self, rhand_anchor.global_position)
				$Sounds/punch_hit.play()
				if "recoil_kick" in active_target:
					active_target.recoil_kick(melee_recoil)
			return
		top_states.DEAD:
			return
	return

func get_target_distance():
	if not active_target:
		return 0.00
	else:
		return self.global_position.distance_to(active_target.global_position)
	return

func get_target_candidates(group_list: Array[String]):
	var candidates: Array[CharacterBody3D] = []
	for group in group_list:
		for actor in get_tree().get_nodes_in_group(group):
			candidates.push_front(actor)
	return candidates

func update_targets_by_distance():
	var target_candidates: Array = get_target_candidates(groups_i_dont_like)
	var lowest_distance: float = -1.0
	var closest_target: CharacterBody3D
	if not target_candidates:
		return
	
	for target in target_candidates:
		if target.is_dead:
			continue
		var target_distance: float = self.global_position.distance_to(target.global_position)
		if target_distance < target_distance_threshold:
			if not target in target_pool:
				target_pool.push_front(target)
			if lowest_distance < 0.0:
				lowest_distance = target_distance
				closest_target = target
				continue
			elif lowest_distance > 0.0 and target_distance < lowest_distance:
				lowest_distance = target_distance
				closest_target = target
				continue
			elif lowest_distance > 0.0 and target_distance > lowest_distance:
				continue
	
	if closest_target:
		active_target = closest_target
	elif not closest_target:
		passive_target = active_target
		active_target = null
	return

func is_target_in_melee_range():
	var target_distance: float = self.global_position.distance_to(active_target.global_position)
	return target_distance and target_distance < melee_distance_threshold

func is_target_viewable():
	if not active_target:
		return
	var sight_y_offset: float = 0.0 if active_target.is_in_group("players") else 5.0
	if (
		active_target and \
		A.is_actor_ahead_approx(self, active_target) and \
		A.is_actor_viewable(self, sight_raycast, active_target, sight_y_offset)
	):
		last_seen_target_position = active_target.global_position
		return true
	else:
		return false
	return

func strike_target(target: Node3D):
	if not active_weapon_node:
		push_warning(self.name + ": Tried to strike without a weapon node.")
		return
	
	is_attacking = true
	attack_cooloff.start()
	anim_h("aim_shoot")
	active_weapon_node.primary_action(self)
	var hit_recipient: Node3D = A.create_impact(self, strike_raycast, active_weapon_node.damage)
	#if is_instance_of(hit_recipient, CharacterBody3D) and hit_recipient.is_dead:
	if hit_recipient and hit_recipient.is_in_group("actors") and hit_recipient.is_dead:
		passive_target = null
		active_target = null
	return

func equip_weapon(weapon_name: String, equip_anim: String = "draw_pistol2"):
	var weapon_item: Dictionary = Weapons.by_name(weapon_name)
	if active_weapon_node:
		active_weapon_node.free()
	
	for c in rhand_anchor.get_children():
		c.free()
	
	is_equipping = true
	equip_delay.start()
	anim_h(equip_anim)
	active_weapon_name = weapon_name
	active_weapon_node = weapon_item["scene"].instantiate()
	active_weapon_node.prop_weapon = true
	rhand_anchor.add_child(active_weapon_node)
	
	active_weapon_ammo_left = weapon_item["mag_size"]
	active_weapon_mag_size = weapon_item["mag_size"]
	return


func anim_h(anim_name: String, is_queued: bool = true, force_anim: bool = false):
	if anim_name == last_anim and not force_anim:
		return
	anim.play(anim_name)
	if is_queued:
		queued_anims.push_back(anim_name)
	last_anim = anim_name
	return

func move_nav(face_direction: bool = true, new_position: Vector3 = Vector3.ZERO):
	if new_position:
		nav_agent.target_position = new_position
	
	var next_position: Vector3 = nav_agent.get_next_path_position()
	if face_direction:
		A.face_position(self, next_position)
		self.velocity = (next_position - self.global_position).normalized() * move_speed
		#A.apply_move(self, Vector2.UP, move_speed)
	elif not face_direction:
		self.velocity = (next_position - self.global_position).normalized() * move_speed
	return

func top_state_switch_to(new_state: int):
#	if self.name == "wiseguy": # DEBUG
#		prints(top_states.keys()[top_state], " -> ", top_states.keys()[new_state])
	previous_top_state = top_state
	top_state = new_state
	return


func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	#if is_dead: return
	
	var closest_hitbox: CollisionShape3D
	if hit_pos:
		var hitbox_positions: Dictionary = {}
		for hitbox in hitboxes:
			hitbox_positions[hitbox.global_position] = hitbox
		
		var closest_position: Vector3 = U.get_closest_position(hit_pos, hitbox_positions.keys())
		closest_hitbox = hitbox_positions[closest_position]
	
	if caller and not is_dead:
		var caller_distance: float = self.global_position.distance_to(caller.global_position)
		if caller_distance > target_distance_threshold:
			target_distance_threshold = caller_distance
		active_target = caller
		#await get_tree().create_timer(0.5).timeout # Feels weird, reimplement better
		A.face_position(self, active_target.global_position)
	
#	if hit_type == U.hit_types.UNDEFINED:
	if is_dead and flinch_timer.is_stopped() and not is_flinching:
		anim_h("dead_flinch2", false, true)
		return
	
	if is_dead:
		return
	
	if closest_hitbox == hitbox_head:
		health -= damage * 3
	else:
		health -= damage
	
	if health > 0:
		flinch_control(closest_hitbox)
		var ouch_sound_index: int = randi_range(0, len(ouch_sounds) - 1)
		ouch_sounds[ouch_sound_index].play()
	elif health < 1:
		kill_actor(closest_hitbox)
	return

func flinch_control(hitbox: CollisionShape3D):
	flinch_timer.start()
	is_flinching = true
	var flinch_anim: String
	match hitbox:
		hitbox_head:
			flinch_anim = "flinch_head"
		hitbox_torso:
			flinch_anim = ["flinch_torso1", "flinch_torso2"][randi_range(0, 1)]
		hitbox_lower:
			flinch_anim = "flinch_lower1"
		hitbox_knees:
			flinch_anim = "flinch_lower1"
		_:
			flinch_anim = "flinch"
	
	anim_h(flinch_anim, false, true)
	return

func kill_actor(hitbox: CollisionShape3D):
	roam_timer.stop()
	if not self.velocity.is_zero_approx():
		var tween: Tween = get_tree().create_tween()
		tween.tween_property(self, "velocity", Vector3.ZERO, 1.0)
	
	if len(die_sounds) > 0:
		var die_sound_index: int = randi_range(0, len(die_sounds) - 1)
		die_sounds[die_sound_index].play()
	top_state_switch_to(top_states.DEAD)
	
	var die_anim: String
	match hitbox:
		hitbox_head:
			die_anim = ["die2", "die_head1", "die_head2"][randi_range(0, 2)]
		hitbox_torso:
			die_anim = "die_torso"
		hitbox_lower:
			die_anim = "die_lower"
		_:
			die_anim = "die1"
	
	anim_h(die_anim, false, true)
	
	is_dead = true
	
	self.remove_from_group("hit_takers")
	
	#for group in ["players", "player_minions"]:
	for group in ["actors"]:
		for p in get_tree().get_nodes_in_group(group):
			self.add_collision_exception_with(p)
	
	for brush in get_tree().get_nodes_in_group("breaktex_brushes"):
		if U.distance(self, brush) < 9:
			brush.activate(self)
			break
	
	drop_weapon()
	$dead_perf_timer.start()
	return

func drop_weapon():
	var world_item: RigidBody3D = Weapons.by_name(spawn_weapon)["world_item"].instantiate()
	Blackboard.current_world.add_child(world_item)
	world_item.global_position = rhand_anchor.global_position
	world_item.rotation_degrees.y = 180
	
	world_item.add_collision_exception_with(self)
	for p in get_tree().get_nodes_in_group("players"):
		world_item.add_collision_exception_with(p) # This shouldn't go here!
	
	world_item.linear_velocity = Vector3(0, 10, 0)
	world_item.angular_velocity = Vector3(0, 0, 5)
	
	for c in rhand_anchor.get_children():
		c.queue_free()
	
	return

func is_roam_chance_triggered():
	if enable_static_roam_chance:
		if static_roam_chance:
			return true
		elif not static_roam_chance:
			return false
			
	elif not enable_static_roam_chance:
		return randf_range(0.0, 0.99) < roam_chance
	return

#############################################


func on_behavior_process_tick():
	if is_dead and top_state == top_states.DEAD:
		return
	behavior_control()
	return

func _on_debug_ticker_timeout():
	return

func _on_attack_warn_timer_timeout():
	is_aiming = false
	top_state_switch_to(top_states.ATTACK)
	return

func _on_reload_timer_timeout():
	return

func _on_alert_fatigue_timeout():
	is_following_sound = false
	top_state_switch_to(top_states.IDLE)
	return

func _on_taunt_timer_timeout():
	return



func _on_equip_delay_timeout():
	is_equipping = false
	return


func _on_attack_cooloff_timeout():
	is_attacking = false
	is_melee_attacking = false
	top_state_switch_to(top_states.ALERT)
	return


func _on_roam_timer_timeout():
	is_roaming = false
	self.velocity = Vector3.ZERO
	roam_position = Vector3.ZERO
	
	if Global.skill_level == Global.skill_levels.PARALYSIS:
		roam_timer.wait_time = randf_range(0.3, 0.9)
	
	top_state_switch_to(top_states.ATTACK_INTENT) # TEMP
	return


func _on_flinch_timer_timeout():
	is_flinching = false
	return


func _on_dead_perf_timer_timeout():
	sight_raycast.enabled = false
	strike_raycast.enabled = false
	return


func _on_melee_warn_timer_timeout():
	is_aiming = false
	top_state_switch_to(top_states.MELEE_ATTACK)
	return


func _world_queue_execute() -> void:
	on_behavior_process_tick()
	#animation_control()
	#hearing_control()
	return

func world_queue_checkin() -> void:
	if world_queue_on_spawn and not is_in_world_queue:
		Blackboard.add_to_universal_world_queue(self)
	return
