@tool
extends EditorScript

var bone_attachment_mappings: Dictionary = {
	"attachment_head": "head",
	"attachment_chest": "chest",
	"attachment_hand_L": "hand.L",
	"attachment_hand_R": "hand.R",
	"attachment_pelvis": "pelvis",
	"attachment_lower_leg_L": "lower_leg.L",
	"attachment_lower_leg_R": "lower_leg.R",
	"attachment_upper_leg_L": "upper_leg.L",
	"attachment_upper_leg_R": "upper_leg.R",
	"attachment_foot_L": "foot.L",
	"attachment_foot_R": "foot.R"
}

var hitbox_names: Array[String] = [
	"hitbox_head",
	"hitbox_chest",
	"hitbox_pelvis",
	"hitbox_lower_leg_L",
	"hitbox_lower_leg_R",
	"hitbox_upper_leg_L",
	"hitbox_upper_leg_R",
	"hitbox_foot_L",
	"hitbox_foot_R"
]

func _run():
	#actor_creator_scratch()
	
	reset_model()
	
	return

func create_nav_agent(actor_root: CharacterBody3D):
	var new_nav_agent: NavigationAgent3D = NavigationAgent3D.new()
	actor_root.add_child(new_nav_agent)
	new_nav_agent.owner = get_scene().get_tree().edited_scene_root
	new_nav_agent.name = "NavigationAgent3D"
	return

func create_raycasts(actor_root: CharacterBody3D):
	var sight_raycast: RayCast3D = RayCast3D.new()
	sight_raycast.name = "sight_raycast"
	var strike_raycast: RayCast3D = RayCast3D.new()
	strike_raycast.name = "strike_raycast"
	
	for rc in [sight_raycast, strike_raycast]:
		actor_root.add_child(rc)
		rc.owner = get_scene().get_tree().edited_scene_root
		var approx_chest_height: float = 5.325
		rc.position.y = approx_chest_height
		rc.target_position.z = -200
	
	return

func create_hitbox_colliders(actor_root: CharacterBody3D, hitbox_names: Array):
	for hitbox_name in hitbox_names:
		var base_name: String = hitbox_name.replace("hitbox_", "")
		var new_hitbox: CollisionShape3D = CollisionShape3D.new()
		new_hitbox.name = hitbox_name
		#new_hitbox.shape = BoxShape3D.new()
		var part_name: String = base_name.replace("_L", "").replace("_R", "")
		var colshape_respath: String = "res://Resources/hitbox_default_" + part_name + ".shape"
		new_hitbox.shape = load(colshape_respath)
		actor_root.add_child(new_hitbox)
		new_hitbox.owner = get_scene().get_tree().edited_scene_root
	return

func create_bone_attachments(skeleton: Skeleton3D, names_and_bones: Dictionary):
	for attachment_name in names_and_bones.keys():
		var new_bone_attachment: BoneAttachment3D = BoneAttachment3D.new()
		new_bone_attachment.name = attachment_name
		new_bone_attachment.bone_name = names_and_bones[attachment_name]
		skeleton.add_child(new_bone_attachment)
		new_bone_attachment.owner = get_scene().get_tree().edited_scene_root
	return

func create_remote_transforms(actor_root: CharacterBody3D, skeleton: Skeleton3D, hitbox_names: Array):
	for hitbox_name in hitbox_names:
		var hitbox_node: CollisionShape3D = actor_root.find_child(hitbox_name)
		var base_name: String = hitbox_name.replace("hitbox_", "")
		var attachment_name: String = "attachment_" + base_name
		var attachment_node: BoneAttachment3D = skeleton.find_child(attachment_name)
		var new_remote_transform: RemoteTransform3D = RemoteTransform3D.new()
		attachment_node.add_child(new_remote_transform)
		new_remote_transform.name = "remote_" + base_name
		if base_name.contains("lower_leg"):
			new_remote_transform.position.y = 0.788
		elif base_name.contains("upper_leg"):
			new_remote_transform.position.y = 0.987
		elif base_name.contains("foot"):
			new_remote_transform.position.y = 0.462
		new_remote_transform.owner = get_scene().get_tree().edited_scene_root
		#new_remote_transform.remote_path = hitbox_node.get_path() # Screwed up, uses an editor path
		new_remote_transform.remote_path = new_remote_transform.get_path_to(hitbox_node)
	return

func create_model(actor_root: CharacterBody3D, respath: String, new_name: String = "", is_flipped: bool = false):
	var new_model: Node3D = load(respath).instantiate()
	actor_root.add_child(new_model)
	new_model.owner = get_scene().get_tree().edited_scene_root
	if new_name:
		new_model.name = new_name
	if is_flipped:
		new_model.rotation_degrees.y = -180
	return new_model

func create_actor_script(actor_root: CharacterBody3D, template_script_respath: String):
	var dir: DirAccess = DirAccess.open("res://Scripts/")
	var new_script_path: String = "res://Scripts/" + actor_root.name + ".gd"
	dir.copy("res://Scripts/actor_characterbody_master.gd", new_script_path)
	actor_root.set_script(load(new_script_path))
	return

func create_spatial_eyes(actor_root: CharacterBody3D):
	var new_mid_anchor: Node3D = Node3D.new()
	new_mid_anchor.name = "mid_anchor"
	new_mid_anchor.position.y = 3.966
	actor_root.add_child(new_mid_anchor)
	new_mid_anchor.owner = get_scene().get_tree().edited_scene_root
	
	var new_mid_eyes: Node3D = Node3D.new()
	new_mid_eyes.name = "eyes"
	new_mid_anchor.add_child(new_mid_eyes)
	new_mid_eyes.owner = get_scene().get_tree().edited_scene_root
	
	
	var new_base_anchor: Node3D = Node3D.new()
	new_base_anchor.name = "base_anchor"
	actor_root.add_child(new_base_anchor)
	new_base_anchor.owner = get_scene().get_tree().edited_scene_root
	
	var new_base_eyes: Node3D = Node3D.new()
	new_base_eyes.name = "eyes"
	new_base_anchor.add_child(new_base_eyes)
	new_base_eyes.owner = get_scene().get_tree().edited_scene_root
	return

func create_debug_label(actor_root: CharacterBody3D):
	var debug_label: Label3D = Label3D.new()
	debug_label.position.y = 7.5
	debug_label.name = "debug_label"
	debug_label.text = "..."
	debug_label.billboard = 1
	debug_label.pixel_size = 0.01
	actor_root.add_child(debug_label)
	debug_label.owner = get_scene().get_tree().edited_scene_root
	return

func assign_groups(actor_root: CharacterBody3D):
	actor_root.add_to_group("hit_takers", true)
	actor_root.add_to_group("actors", true)
	return

func actor_creator_scratch():
	var model_respath: String = "res://Staging/Models/punk_male.glb"
	var template_script_respath: String = "res://Scripts/actor_characterbody_master.gd"
	var actor: CharacterBody3D = get_scene()
	var model: Node3D = create_model(actor, model_respath, "model", true)
	var skel: Skeleton3D = model.find_child("Skeleton3D")
	
	get_scene().set_editable_instance(model, true)
	
	create_hitbox_colliders(actor, hitbox_names)
	create_bone_attachments(skel, bone_attachment_mappings)
	create_remote_transforms(actor, skel, hitbox_names)
	create_actor_script(actor, template_script_respath)
	create_spatial_eyes(actor)
	create_raycasts(actor)
	create_nav_agent(actor)
	create_debug_label(actor)
	assign_groups(actor)
	return

func reset_model():
	var model_respath: String = get_scene().get_node("model").scene_file_path
	var actor: CharacterBody3D = get_scene()
	
	# Delete model and hitboxes
	get_scene().get_node("model").free()
	for hitbox in get_scene().find_children("hitbox_*"):
		hitbox.free()
	
	var model: Node3D = create_model(actor, model_respath, "model", true)
	var skel: Skeleton3D = model.find_child("Skeleton3D")
	
	get_scene().set_editable_instance(model, true)
	
	create_hitbox_colliders(actor, hitbox_names)
	create_bone_attachments(skel, bone_attachment_mappings)
	create_remote_transforms(actor, skel, hitbox_names)
	
	return
