@tool
extends StaticBody3D

@onready var light_anim: AnimationPlayer = $button_light_anchor/button_light/AnimationPlayer
@onready var buttons: Array = $Meshes.find_children("button_*", "MeshInstance3D")
@onready var button_light: SpotLight3D = $button_light_anchor/button_light
@onready var code_light: OmniLight3D = $code_lights/colored_light
@onready var hurt_light: OmniLight3D = $hurt_light
@onready var button_light_anchor: Node3D = $button_light_anchor
@onready var button_light_marker: Marker3D = $button_light_anchor/button_light/Marker3D
var button_light_atten_on: float = 5.46
var button_light_atten_off: float = 1.00
var button_light_row_offset: float = .365
var button_light_col_offset: float = .27
#var button_light_hbounds: Vector2 = Vector2()
#var button_light_vbounds: Vector2 = Vector2()

@export var health: int = 250
var starting_health: int = health

@export var angles: Vector3 = Vector3.ZERO
@export var rot_from_angles: bool = true
@export var passcode: String = ""
@export var target_group: String = ""
@export var oneshot: bool = false

var interaction_distance: float = 36.0
var is_being_pressed: bool = false
var tick_interval: float = 0.1 # .1
var ticker: float = 0.0
var user: Node3D
var focused_button: MeshInstance3D
var last_pressed_button: MeshInstance3D
var last_user_rotation: Vector3 = Vector3.ZERO

var user_entered_code: String = ""
var is_previously_activated: bool = false
var is_glitching: bool = false
var is_dead: bool = false


func _enter_tree() -> void:
	if Engine.is_editor_hint():
		if rot_from_angles:
			self.rotation_degrees = angles
		
		var removals: Array = []
		for m in self.get_children():
			if m.name.contains("entity_"):
				removals.append(m)
				continue
			if m.name == "CollisionShape3D":
				removals.append(m)
				continue
		
		for r in removals:
			r.queue_free()
		return
	else:
		return
	return

func _ready() -> void:
	if Engine.is_editor_hint():
		return
	else:
		starting_health = health
		return
	return

func _physics_process(delta: float) -> void:
	if Engine.is_editor_hint():
		return
	
	ticker += delta
	if ticker >= tick_interval:
		ticker = 0.0
		tick_process()
	
	if is_glitching:
		#$Sounds/code_good.pitch_scale = randf_range(0.3, 1.5)
		$Sounds/code_good.pitch_scale = [1.0, .7, 1.1, .2].pick_random()
	
	return

func dial_num(button_number: int = 0) -> void:
	if button_number == 0:
		return
	if is_being_pressed:
		return
	is_being_pressed = true
	var button_mesh: MeshInstance3D = self.get_node("Meshes/button_" + str(button_number))
	
	$Sounds/press.play()
	var base_seq_duration: float = 0.1 # 0.23 # 0.1
	get_tree().create_tween().tween_property(
		button_mesh,
		"position:x",
		-0.09,
		base_seq_duration
	).as_relative()
	await get_tree().create_timer(base_seq_duration).timeout
	get_tree().create_tween().tween_property(
		button_mesh,
		"position:x",
		0.09,
		base_seq_duration * 2
	).as_relative().set_trans(Tween.TRANS_BACK)
	await get_tree().create_timer((base_seq_duration * 2)).timeout
	is_being_pressed = false
	return

func push_button(button_mesh: MeshInstance3D) -> void:
	#if is_being_pressed:
	if is_being_pressed and last_pressed_button == button_mesh:
		return
	
	is_being_pressed = true
	last_pressed_button = button_mesh
	$Sounds/press.pitch_scale = randf_range(.9, 1.2)
	$Sounds/press.play()
	var base_seq_duration: float = 0.1 # 0.23 # 0.1
	get_tree().create_tween().tween_property(
		button_mesh,
		"position:x",
		-0.09,
		base_seq_duration
	).as_relative()
	await get_tree().create_timer(base_seq_duration).timeout
	get_tree().create_tween().tween_property(
		button_mesh,
		"position:x",
		0.09,
		base_seq_duration * 2
	).as_relative().set_trans(Tween.TRANS_BACK)
	await get_tree().create_timer((base_seq_duration * 2)).timeout
	is_being_pressed = false
	return

func interact(caller: Object) -> void:
	# dial_num(5) # DEBUG
	if not focused_button:
		prints(self.name, "Interacted with no focused_button:", focused_button)
		return
	
	push_button(focused_button)
	
	#if oneshot and is_previously_activated:
		##play_code_rejected()
		#return
	#
	code_entry(focused_button.name.right(1))
	return

func tick_process() -> void:
	var player: Object = get_tree().get_first_node_in_group("players")
	if not player:
		return
	
	
	var player_sqdist: float = self.global_position.distance_squared_to(player.global_position)
	
	if player_sqdist <= interaction_distance:
		if user == null:
			user = player
			if light_anim.assigned_animation != "focus":
				light_anim.play("focus")
				button_light.spot_attenuation = button_light_atten_on
		if not self in user.passive_input_receivers:
			user.passive_input_receivers.append(self)
		proximity_interaction_control()
	elif player_sqdist > interaction_distance:
		if user == player:
			#if self in user.passive_input_receivers:
			user.passive_input_receivers.erase(self)
			user = null
			focused_button = null
			user_entered_code = ""
			button_light.position = Vector3.ZERO
			if light_anim.assigned_animation == "focus":
				button_light.spot_attenuation = button_light_atten_off
				light_anim.play_backwards("focus")
				light_anim.queue("start")
	return

func proximity_interaction_control() -> void:
	if not user:
		prints(self.name, "Called proximity_interaction_control with no user:", user)
		return
	
	
	
	# Tried to use rotation difference but doing float math or vector math returned 0, I guess it was too close (within ~.001) and rounded. Whatever!
	#var rot_diff: Vector3 = user.global_rotation_degrees - last_user_rotation
	#var move_scale: float = .1
	#button_light.position.x += rot_diff.y * move_scale
	#button_light.position.y += rot_diff.x * move_scale
	
	var closest_button: MeshInstance3D
	var closest_dist: float = -1.0
	for button in buttons:
		var button_dist: float = button_light_marker.global_position.distance_squared_to(button.global_position)
		if closest_dist < 0.0:
			closest_button = button
			closest_dist = button_dist
			continue
		if button_dist < closest_dist:
			closest_button = button
			closest_dist = button_dist
			continue
	if closest_dist < 0.0:
		prints(self.name, "Problem? Closest dist is still < 0.0 after checking buttons.")
		return
	
	focused_button = closest_button
	
	return

func _user_input_event_push(caller: Object, event: InputEvent) -> void:
	if not event is InputEventMouseMotion:
		return
	
	var move_scale: float = .008

	button_light.position.x += event.relative.x * move_scale
	# .45
	button_light.position.y += -event.relative.y * move_scale

	# .33
	
	button_light.position.x = clampf(
		button_light.position.x,
		-0.33,
		0.33
	)
	# .45
	button_light.position.y = clampf(
		button_light.position.y,
		-.45,
		.45
	)
	return

func code_entry(numstring: String) -> void:
	#code_entry(focused_button.name.right(1))
	# 4 digits
	# If the code is 4 digits after appending, trigger submit
	user_entered_code += numstring
	if $debug_label.visible:
		$debug_label.text = user_entered_code
	if len(user_entered_code) >= 4:
		code_submit()
		return
	return

func code_submit() -> void:
	# If code is correct, act like it (positive sound, "granted" voice, green light)
	# If code is incorrect, act like it (neg sound, "denied" voice, red light)
	if user_entered_code == passcode:
		is_previously_activated = true
		play_code_accepted()
		user_entered_code = ""
		if target_group:
			if target_group.contains(","):
				var group_list: Array = target_group.split(",")
				for group in group_list:
					for n in get_tree().get_nodes_in_group(group):
						if n == self:
							continue
						if n.has_method("activate"):
							n.activate()
			else:
				for n in get_tree().get_nodes_in_group(target_group):
					if n == self:
						continue
					if n.has_method("activate"):
						n.activate()
		return
	
	if user_entered_code != passcode:
		play_code_rejected()
		user_entered_code = ""
		return
	
	return

func play_code_accepted(delay: float = 0.2) -> void:
	if delay:
		await get_tree().create_timer(delay).timeout
	
	var code_sound: AudioStreamPlayer3D = $Sounds/code_good
	code_sound.play()
	
	if health > 1:
		blink_code_light(Color.GREEN)
	else:
		blink_code_light(Color.RED)
	return

func play_code_rejected(delay: float = 0.2) -> void:
	if delay:
		await get_tree().create_timer(delay).timeout
	var code_sound: AudioStreamPlayer3D = $Sounds/code_bad
	code_sound.play()
	
	blink_code_light(Color.RED)
	return

func blink_code_light(light_color: Color = Color.WHITE, blink_time: float = .17) -> void:
	code_light.visible = false
	var prev_light_color: Color = code_light.light_color
	code_light.light_color = light_color
	
	code_light.visible = true
	await get_tree().create_timer(blink_time).timeout
	code_light.visible = false
	await get_tree().create_timer(blink_time).timeout
	code_light.visible = true
	await get_tree().create_timer(blink_time).timeout
	code_light.visible = false
	
	code_light.light_color = prev_light_color
	return

func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
	if is_dead:
		return
	
	if U.coin_flip() and U.coin_flip():
		$Sounds.find_children("hurt_*").pick_random().play()
	
	health -= damage
	
	$spark_emitter.emitting = true
	$spark_emitter.restart()
	
	var hurt_ratio: float = float(health) / float(starting_health)
	#if hurt_ratio < .50:
	if hurt_ratio < .80:
		$smoke_emitter.emitting = true
	hurt_light.light_energy = lerpf(1.0, 0.0, hurt_ratio)
	$smoke_emitter.amount_ratio = lerpf(1.0, 0.0, hurt_ratio)
	
	if health < 1:
		kill_codepad()
	return

func kill_codepad() -> void:
	if is_dead:
		return
	is_dead = true
	oneshot = true
	is_glitching = true
	
	$Sounds/die_1.play()
	
	var glitch_time: float = 2.2
	user_entered_code = passcode
	code_submit()
	await get_tree().create_timer(glitch_time).timeout
	
	return
