extends Node3D

@export var debug_transparency: bool = false
@export var performance_mode: bool = true
@export var angles: Vector3
@export var alpha_texture: bool
@export var is_floor: bool
@export var hard_floor: bool = false
@export var target_group: String = ""
@export var spawn_disabled: bool
@export var spawn_static: bool = false
@onready var mesh_instance: MeshInstance3D = self.find_children("*", "MeshInstance3D")[0]
@onready var anchor: Marker3D = $anchor
var glass_cell_master: PackedScene = preload("res://Scenes/glass_cell.tscn")
var glass_cell_master_staticbody: PackedScene = preload("res://Scenes/glass_cell_staticbody.tscn")
var cells: Array = []
var bottom_indices: Array = [0, 1, 4, 5]
var cell_size: float = 1.0 # 1.0
var is_ever_impacted: bool = false

var grid_map: Array = [
	# [col_0, col_1, ...] # row_0
	# [col_0, col_1, ...] # row_1
	####
	# grid_map[row_index][column_index] # Output: cell_index
]

func _ready() -> void:
	if debug_transparency:
		for m in self.find_children("*", "MeshInstance3D"):
			m.transparency = .6
	
	#if alpha_texture:
		#U.set_mesh_materials_alpha(mesh_instance)
	
	if is_floor:
		generate_cells_floor(mesh_instance)
	else:
		generate_cells(mesh_instance)
	
	if performance_mode:
		for cell in cells:
			cell.disable_mode = CollisionObject3D.DISABLE_MODE_MAKE_STATIC
			cell.process_mode = Node.PROCESS_MODE_DISABLED
	
	if spawn_disabled:
		for cell in cells:
			#cell.process_mode = Node.PROCESS_MODE_DISABLED
			cell.toggle_colliders(false)
	return

func generate_cells(host_mesh_instance: MeshInstance3D) -> void:
	var endpoints: Dictionary = U.get_aabb_global_endpoints_rich(host_mesh_instance)
	anchor.global_position = endpoints[0]
	anchor.rotation_degrees.y = angles.y
	var mesh_aabb: AABB = host_mesh_instance.mesh.get_aabb()
	var grid_height: float = mesh_aabb.size.y
	var grid_width: float = mesh_aabb.size.x if mesh_aabb.size.x > mesh_aabb.size.z else mesh_aabb.size.z
	
	var row_count: int = int(grid_height / cell_size)
	var col_count: int = int(grid_width / cell_size)
	var row_index: int = -1
	var col_index: int = -1
	for row in range(row_count):
		#row_index += 1
		row_index = row
		grid_map.insert(row_index, [])
		for col in range(col_count):
			#col_index += 1
			col_index = col
			var new_cell: PhysicsBody3D
			if spawn_static:
				new_cell = glass_cell_master_staticbody.instantiate()
			elif not spawn_static:
				new_cell = glass_cell_master.instantiate()
			var next_cell_index: int = cells.size()
			grid_map[row_index].insert(col_index, next_cell_index)
			cells.append(new_cell)
			#self.add_child(new_cell)
			
			if is_instance_of(new_cell, RigidBody3D):
				new_cell.process_mode = Node.PROCESS_MODE_DISABLED
			
			anchor.add_child(new_cell)
			#new_cell.rotation_degrees.y = angles.y
			#new_cell.global_position = endpoints[0]
			new_cell.global_position = endpoints[0].move_toward(endpoints[1], .5)
			
			#new_cell.position.y += (row * cell_size) + 0.5
			#new_cell.position.x += (col * cell_size)
			
			var x_offset: float = .5 # * (1.0 + (absf(angles.y) / 90))
			var add_vec: Vector3 = Vector3.ZERO
			add_vec.y += (row * cell_size) + 0.5
			add_vec.x += (col * cell_size) + x_offset
			
			new_cell.position += add_vec
			new_cell.original_origin = new_cell.global_position
			new_cell.is_in_grid = true
			new_cell.visible = false
			#U.set_model_material(new_cell, mesh_instance.get_active_material(0), 0.99)
			# DEBUG # new_cell.process_mode = Node.PROCESS_MODE_DISABLED
	
	await get_tree().process_frame
	for cell in cells:
		cell.process_mode = Node.PROCESS_MODE_INHERIT
	
	return


func generate_cells_floor(host_mesh_instance: MeshInstance3D) -> void:
	var endpoints: Dictionary = U.get_aabb_global_endpoints_rich(host_mesh_instance)
	anchor.global_position = endpoints[0]
	anchor.rotation_degrees.y = angles.y
	var mesh_aabb: AABB = host_mesh_instance.mesh.get_aabb()
	#var grid_height: float = mesh_aabb.size.y # Window-Wall
	var grid_height: float = mesh_aabb.size.z # Window-Floor
	var grid_width: float = mesh_aabb.size.x if mesh_aabb.size.x > mesh_aabb.size.y else mesh_aabb.size.y
	
	var row_count: int = int(grid_height / cell_size)
	var col_count: int = int(grid_width / cell_size)
	var row_index: int = -1
	var col_index: int = -1
	for row in range(row_count):
		#row_index += 1
		row_index = row
		grid_map.insert(row_index, [])
		for col in range(col_count):
			#col_index += 1
			col_index = col
			var new_cell: PhysicsBody3D
			if spawn_static:
				new_cell = glass_cell_master_staticbody.instantiate()
			elif not spawn_static:
				new_cell = glass_cell_master.instantiate()
			
			var next_cell_index: int = cells.size()
			grid_map[row_index].insert(col_index, next_cell_index)
			cells.append(new_cell)
			new_cell.process_mode = Node.PROCESS_MODE_DISABLED
			anchor.add_child(new_cell)
			new_cell.global_position = endpoints[0].move_toward(endpoints[1], .5)
			new_cell.rotation_degrees.x = 90.0
			
			#new_cell.position.y += (row * cell_size) + 0.5
			#new_cell.position.x += (col * cell_size)
			
			var x_offset: float = .5 # * (1.0 + (absf(angles.y) / 90))
			var add_vec: Vector3 = Vector3.ZERO
			add_vec.z += (row * cell_size) + 0.5
			add_vec.x += (col * cell_size) + x_offset
			
			new_cell.position += add_vec
			new_cell.original_origin = new_cell.global_position
			new_cell.is_in_grid = true
			new_cell.visible = false
			#U.set_model_material(new_cell, mesh_instance.get_active_material(0), 0.99)
			# DEBUG # new_cell.process_mode = Node.PROCESS_MODE_DISABLED
	
	await get_tree().process_frame
	for cell in cells:
		cell.process_mode = Node.PROCESS_MODE_INHERIT
	prints(self.name, "Cell generation completed. Grid map:", grid_map)
	return


func _cell_impacted(impacted_cell: PhysicsBody3D) -> void:
	if U.coin_flip() and U.coin_flip() and is_instance_of(impacted_cell, StaticBody3D):
		cellstatic_impact_splash(impacted_cell)
	
	if is_ever_impacted:
		return
	
	is_ever_impacted = true
	mesh_instance.visible = false
	var cells_arrproxy: Array = cells
	for cell in cells_arrproxy:
		if not cell:
			continue
		cell.visible = true
		#if is_floor and not hard_floor and is_instance_of(cell, StaticBody3D):
	
	####

	return

func activate(caller: Object = null) -> void:
	for cell in cells:
		#cell.process_mode = Node.PROCESS_MODE_INHERIT
		cell.toggle_colliders(true)
	return

func get_cell_index(cell_node: PhysicsBody3D) -> int:
	return cells.find(cell_node)

func get_gridmap_coordinates(cell_index: int) -> Vector2:
	var unused_vector2: Vector2 = Vector2(-1, -1)
	var cell_coordinates: Vector2 = unused_vector2
	var grid_map_last_row: int = grid_map.size() - 1
	for row in range(grid_map_last_row):
		if cell_coordinates != unused_vector2:
			break
		var last_col_of_row: int = grid_map[row].size() - 1
		for col in range(last_col_of_row):
			if grid_map[row][col] == cell_index:
				cell_coordinates = Vector2(row, col)
				break
	return cell_coordinates

func get_cell_index_in_gridmap(coordinates: Vector2 = Vector2(-1, -1)) -> int:
	var cell_index: int = -1
	if coordinates == Vector2(-1, -1):
		return cell_index
	
	cell_index = grid_map[coordinates.x][coordinates.y]
	return cell_index

func cellstatic_impact_splash(impacted_cell: PhysicsBody3D) -> void:
	# Break a row beneath
	var cell_coordinates: Vector2 = get_gridmap_coordinates(get_cell_index(impacted_cell))
	#if U.coin_flip() and U.coin_flip():
	var last_col: int = grid_map[cell_coordinates.x].size() - 1
	var col_wiggle: int = [1, -1].pick_random()
	var deviant_col: int = cell_coordinates.y + col_wiggle
	if deviant_col <= last_col and deviant_col >= 0:
		cell_coordinates.y = deviant_col
	
	
	var last_row: int = grid_map.size() - 1
	
	## \/ This I think was getting kind of recursive. It would loop through all ranged rows, then on those cells it would loop through all ranged rows
	# (because glassgrid would call cellstatic_impact_splash on each impact, duh
	
	#for row in range(cell_coordinates.x, grid_map.size(), 1): # I thought this would go down but the rows start from the bottom
		#var splash_cell_index: int = grid_map[row][cell_coordinates.y]
		##func hit(damage: int, hit_type: int, caller: PhysicsBody3D, hit_pos: Vector3):
		#var splash_cell: PhysicsBody3D = cells[splash_cell_index]
		#if splash_cell.is_ever_impacted:
			#continue
		#splash_cell.hit(1, U.hit_types.AREA, null, splash_cell.global_position)
	
	## /\
	
	var next_row_down: int = cell_coordinates.x - 1
	if next_row_down < 0:
		return
	
	var splash_cell_index: int = grid_map[next_row_down][cell_coordinates.y]
	if not is_instance_valid(cells[splash_cell_index]):
		return
	var splash_cell: PhysicsBody3D = cells[splash_cell_index]
	if splash_cell.is_ever_impacted:
		return
	
	splash_cell.hit(1, U.hit_types.AREA, null, splash_cell.global_position)
	return


func _on_long_tick_timeout() -> void:
	if is_ever_impacted and performance_mode:
		separate_random_cell()
		$long_tick.wait_time = 1.2 * randf_range(.1, 1.0)
	return

func separate_random_cell() -> void:
	var cell_candidates: Array = cells.filter(
		func (cell):
			if not is_instance_valid(cell):
				return false
			if not cell.is_separated and not cell.is_ever_impacted:
				return true
	)
	var cell_choice: Object = cell_candidates.pick_random()
	if cell_choice == null:
		return
	else:
		cell_choice.set_as_separated()
	return
